/**
 * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/
 * @author     Mediotype                     https://www.mediotype.com/
 */

define([
    'jquery',
    'uiRegistry',
    'Magento_Checkout/js/checkout-data'
], function ($, registry, checkoutData) {
    'use strict';

    return function (Component) {
        return Component.extend({
            validationConfig: window.checkoutConfig.vertexAddressValidationConfig,
            addressValidator: null,

            /**
             * @returns {Object}
             */
            initialize: function () {
                const self = this;
                this._super();

                registry.get(
                    'checkout.steps.billing-step.payment.payments-list' +
                    '.before-place-order.billingAdditional' +
                    '.address-validation-message.validator',
                    function (validator) {
                        this.addressValidator = validator;
                    }.bind(this)
                );

                this.isAddressDetailsVisible.subscribe(function (isVisible) {
                    self.addressDetailsVisibilityChanged(isVisible);
                });

                return this;
            },

            /**
             * @returns {self}
             */
            updateAddress: function () {
                this.registerAddressInvalidationTrigger();

                const billingData = checkoutData.getBillingAddressFromData();

                // jscs:disable requireCamelCaseOrUpperCaseIdentifiers
                if (!this.validationConfig.isAddressValidationEnabled ||
                    this.addressValidator.isAddressValid ||
                    billingData === null ||
                    this.selectedAddress() && !this.isAddressFormVisible() ||
                    this.validationConfig.countryValidation.indexOf(billingData.country_id) === -1
                ) {
                    return this._super();
                }
                // jscs:enable requireCamelCaseOrUpperCaseIdentifiers

                this.addressValidator.addressValidation().done(function () {
                    if (!this.validationConfig.showValidationSuccessMessage) {
                        return this.updateAddress();
                    }
                }.bind(this));
            },

            /**
             * When called, register a single (mind the "one") address invalidation trigger,
             * that sets the "this.addressValidator.isAddressValid = false" for any further billing address field change.
             */
            registerAddressInvalidationTrigger: function () {
                let that = this;
                $('fieldset')
                    .find('[data-form="billing-new-address"]')
                    .one(
                        'keyup change paste',
                        'input[name^="street"]' +
                        ', input[name="postcode"]' +
                        ', input[name="city"]' +
                        ', input[name="country_id"]' +
                        ', select[name="region_id"]',
                        function () {
                            that.addressValidator.isAddressValid = false;
                        });
            },

            /**
             * If the address details are visible, then remove the validation address warning message
             *
             * @param {Boolean} isVisible
             */
            addressDetailsVisibilityChanged: function (isVisible) {
                let message = this.addressValidator.message;

                if (isVisible && message.hasMessage() && message.message.type === 1) {
                    message.clear();
                }
            }
        });
    };
});
