<?php
/**
 * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/
 * @author     Mediotype                     https://www.mediotype.com/
 */

namespace Vertex\Tax\Model;

use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Message\ManagerInterface;
use Magento\Framework\Phrase;
use Magento\Sales\Api\OrderStatusHistoryRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Creditmemo;
use Magento\Sales\Model\Order\Creditmemo\Item as CreditmemoItem;
use Magento\Sales\Model\Order\Invoice;
use Magento\Sales\Model\Order\Invoice\Item as InvoiceItem;
use Magento\Sales\Model\Order\Item as OrderItem;
use Magento\Store\Model\ScopeInterface;
use Psr\Log\LoggerInterface;
use Vertex\Exception\ApiException;
use Vertex\Services\Invoice\RequestInterface;
use Vertex\Services\Invoice\RequestInterfaceFactory;
use Vertex\Tax\Api\InvoiceInterface;
use Vertex\Tax\Model\Api\Data\CustomerBuilder;
use Vertex\Tax\Model\Api\Data\SellerBuilder;
use Vertex\Tax\Model\TaxInvoice\BaseFormatter;
use Vertex\Tax\Model\TaxInvoice\ItemFormatter;

/**
 * Service for sending Tax Invoices to the Vertex API
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class TaxInvoice
{
    const REQUEST_TYPE = 'invoice';
    const REQUEST_TYPE_REFUND = 'invoice_refund';

    /** @var BaseFormatter */
    private $baseFormatter;

    /** @var Config */
    private $config;

    /** @var CustomerBuilder */
    private $customerBuilder;

    /** @var DateTimeImmutableFactory */
    private $dateTimeFactory;

    /** @var InvoiceInterface */
    private $invoice;

    /** @var ItemFormatter */
    private $itemFormatter;

    /** @var LoggerInterface */
    private $logger;

    /** @var ManagerInterface */
    private $messageManager;

    /** @var OrderStatusHistoryRepositoryInterface */
    private $orderStatusRepository;

    /** @var RequestInterfaceFactory */
    private $requestFactory;

    /** @var SellerBuilder */
    private $sellerBuilder;

    /**
     * @param LoggerInterface $logger
     * @param ManagerInterface $messageManager
     * @param CustomerBuilder $customerBuilder
     * @param OrderStatusHistoryRepositoryInterface $orderStatusRepository
     * @param BaseFormatter $baseFormatter
     * @param ItemFormatter $itemFormatter
     * @param RequestInterfaceFactory $requestFactory
     * @param SellerBuilder $sellerBuilder
     * @param Config $config
     * @param InvoiceInterface $invoice
     * @param DateTimeImmutableFactory $dateTimeFactory
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        LoggerInterface $logger,
        ManagerInterface $messageManager,
        CustomerBuilder $customerBuilder,
        OrderStatusHistoryRepositoryInterface $orderStatusRepository,
        BaseFormatter $baseFormatter,
        ItemFormatter $itemFormatter,
        RequestInterfaceFactory $requestFactory,
        SellerBuilder $sellerBuilder,
        Config $config,
        InvoiceInterface $invoice,
        DateTimeImmutableFactory $dateTimeFactory
    ) {
        $this->logger = $logger;
        $this->messageManager = $messageManager;
        $this->customerBuilder = $customerBuilder;
        $this->orderStatusRepository = $orderStatusRepository;
        $this->baseFormatter = $baseFormatter;
        $this->itemFormatter = $itemFormatter;
        $this->requestFactory = $requestFactory;
        $this->sellerBuilder = $sellerBuilder;
        $this->config = $config;
        $this->invoice = $invoice;
        $this->dateTimeFactory = $dateTimeFactory;
    }

    /**
     * Prepare the data for the Invoice Request
     *
     * @param Order|Invoice|Creditmemo $entityItem
     * @param string|null $event
     * @return RequestInterface
     */
    public function prepareInvoiceData($entityItem, $event = null)
    {
        $order = $entityItem;
        $typeId = 'ordered';

        if ($entityItem instanceof Invoice || $entityItem instanceof Creditmemo) {
            $typeId = 'invoiced';
            $order = $entityItem->getOrder();
        }

        $storeCode = $order->getStoreId();

        $seller = $this->sellerBuilder
            ->setScopeCode($storeCode)
            ->setScopeType(ScopeInterface::SCOPE_STORE)
            ->build();

        /** @var RequestInterface $request */
        $request = $this->requestFactory->create();
        $request->setTransactionType(RequestInterface::TRANSACTION_TYPE_SALE);
        $request->setDocumentNumber($order->getIncrementId());
        $request->setDocumentDate($this->dateTimeFactory->create($order->getCreatedAt()));
        $request->setSeller($seller);
        $request->setCustomer($this->customerBuilder->buildFromOrder($order));
        if ($this->config->getLocationCode($storeCode)) {
            $request->setLocationCode($this->config->getLocationCode($storeCode));
        }

        $orderItems = [];
        $orderedItems = $entityItem->getAllItems();
        foreach ($orderedItems as $item) {
            /** @var OrderItem|InvoiceItem|CreditmemoItem $item */
            $this->itemFormatter->addPreparedOrderItems($orderItems, $typeId, $event, $item);
        }

        $shippingInfo = $this->baseFormatter->getFormattedShippingData($entityItem, $event);
        if (!empty($shippingInfo) && !$order->getIsVirtual()) {
            $orderItems[] = $shippingInfo;
        }

        $orderItems = $this->baseFormatter->addRefundAdjustments($orderItems, $entityItem);
        $orderItems = $this->addIfNotEmpty(
            $orderItems,
            $this->baseFormatter->getFormattedOrderGiftWrap($order, $entityItem, $event),
            $this->baseFormatter->getFormattedOrderPrintCard($order, $entityItem, $event)
        );

        $request->setLineItems($orderItems);
        return $request;
    }

    /**
     * Send the Invoice Request to the API
     *
     * @param RequestInterface $request
     * @param Order $order
     * @return bool
     */
    public function sendInvoiceRequest(RequestInterface $request, Order $order)
    {
        try {
            $response = $this->invoice->record($request, $order->getStoreId(), ScopeInterface::SCOPE_STORE);
        } catch (\Exception $e) {
            $this->addErrorMessage(__('Could not submit invoice to Vertex.'), $e);
            return false;
        }

        $totalTax = $response->getTotalTax();

        $comment = $order->addStatusHistoryComment(
            'Vertex Invoice sent successfully. Amount: $' . number_format($totalTax, 2)
        );
        try {
            $this->orderStatusRepository->save($comment);
        } catch (\Exception $originalException) {
            $exception = new \Exception('Could not save Vertex invoice comment', 0, $originalException);
            $this->logger->critical(
                $exception->getMessage() . PHP_EOL . $exception->getTraceAsString()
            );
        }
        return true;
    }

    /**
     * Send the Creditmemo Request to the API
     *
     * @param RequestInterface $request
     * @param Order|null $order
     * @return bool
     */
    public function sendRefundRequest(RequestInterface $request, Order $order)
    {
        try {
            $response = $this->invoice->record($request, $order->getStoreId(), ScopeInterface::SCOPE_STORE);
        } catch (\Exception $e) {
            $this->addErrorMessage(__('Could not submit refund to Vertex.'), $e);
            return false;
        }

        $totalTax = $response->getTotalTax();

        $comment = $order->addStatusHistoryComment(
            'Vertex Invoice refunded successfully. Amount: $' . number_format($totalTax, 2)
        );
        try {
            $this->orderStatusRepository->save($comment);
        } catch (\Exception $originalException) {
            $exception = new CouldNotSaveException(
                __('Could not save Vertex invoice refund comment'),
                $originalException
            );
            $this->logger->critical(
                $exception->getMessage() . PHP_EOL . $exception->getTraceAsString()
            );
        }
        return true;
    }

    private function addErrorMessage(Phrase $friendlyPhrase, $exception = null)
    {
        $friendlyMessage = $friendlyPhrase->render();
        $errorMessage = null;

        $exceptionClass = get_class($exception);

        switch ($exceptionClass) {
            case ApiException\AuthenticationException::class:
                $errorMessage = __(
                    '%1 Please verify your configured Company Code and Trusted ID are correct.',
                    $friendlyMessage
                );
                break;
            case ApiException\ConnectionFailureException::class:
                $errorMessage = __(
                    '%1 Vertex could not be reached. Please verify your configuration.',
                    $friendlyMessage
                );
                break;
            case ApiException::class:
            default:
                $errorMessage = __('%1 Error has been logged.', $friendlyMessage);
                break;
        }

        $this->messageManager->addErrorMessage($errorMessage);
    }

    /**
     * Append items to the provided array if they are not empty
     *
     * @param array $array
     * @param mixed ...$items
     * @return array
     */
    private function addIfNotEmpty(array $array)
    {
        $items = array_slice(func_get_args(), 1);

        foreach ($items as $item) {
            if (!empty($item)) {
                $array[] = $item;
            }
        }

        return $array;
    }
}
