<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace PayPal\Braintree\Model\Adapter\PaymentMethod;

use Braintree\GooglePayCard;
use DateInterval;
use DateTimeZone;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Intl\DateTimeFactory;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Vault\Api\Data\PaymentTokenFactoryInterface;
use PayPal\Braintree\Gateway\Config\Config;
use PayPal\Braintree\Model\GooglePay\Ui\ConfigProvider;

class BraintreeGooglePayPaymentTokenAdapter implements PaymentTokenAdapterInterface
{
    /**
     * @param GooglePayCard $paymentMethod
     * @param DateTimeFactory $dateTimeFactory
     * @param SerializerInterface $serializer
     * @param Config $config
     */
    public function __construct(
        private readonly GooglePayCard $paymentMethod,
        private readonly DateTimeFactory $dateTimeFactory,
        private readonly SerializerInterface $serializer,
        private readonly Config $config
    ) {
    }

    /**
     * Get the payment method code.
     *
     * @return string
     */
    public function getPaymentMethodCode(): string
    {
        return ConfigProvider::METHOD_CODE;
    }

    /**
     * Get Payment token type.
     *
     * @return string
     */
    public function getType(): string
    {
        return PaymentTokenFactoryInterface::TOKEN_TYPE_CREDIT_CARD;
    }

    /**
     * Get Gateway token.
     *
     * @return string
     */
    public function getGatewayToken(): string
    {
        return $this->paymentMethod->token;
    }

    /**
     * Get the token expiration date.
     *
     * @return string
     */
    public function getExpiresAt(): string
    {
        $expiryDate = $this->dateTimeFactory->create(
            sprintf(
                '%s-%s-01 00:00:00',
                $this->paymentMethod->expirationYear,
                $this->paymentMethod->expirationMonth
            ),
            new DateTimeZone('UTC')
        );

        $expiryDate->add(new DateInterval('P1M'));

        return $expiryDate->format('Y-m-d 00:00:00');
    }

    /**
     * Get the token details.
     *
     * @return string
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function getTokenDetails(): string
    {
        return $this->serializer->serialize([
            'customerId' => $this->paymentMethod->customerId,
            'type' => $this->getCreditCardType($this->paymentMethod->cardType),
            'maskedCC' => $this->paymentMethod->sourceCardLast4,
            'expirationDate' => $this->paymentMethod->expirationMonth . '/' . $this->paymentMethod->expirationYear,
        ]);
    }

    /**
     * Get type of credit card mapped from Braintree.
     *
     * @param string $type
     * @return string
     * @throws InputException
     * @throws NoSuchEntityException
     */
    private function getCreditCardType(string $type): string
    {
        $replaced = str_replace(' ', '-', strtolower($type));
        $mapper = $this->config->getCcTypesMapper();

        return $mapper[$replaced];
    }
}
