<?php
/**
 * Copyright 2025 Adobe
 * All Rights Reserved.
 */

declare(strict_types=1);

namespace PayPal\Braintree\Block\ApplePay\Shortcut;

use Magento\Catalog\Model\Product;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\GroupedProduct\Model\Product\Type\Grouped;

/**
 * @api
 * @since 100.0.2
 */
class ProductPage extends Button
{
    /**
     * @inheritdoc
     */
    public function isActive(): bool
    {
        return $this->applePayConfig->isActive()
            && $this->applePayConfig->displayApplePayButtonOnPdp()
            && ($this->getAmount() > 0);
    }

    /**
     * @inheritdoc
     */
    public function getAlias(): string
    {
        return 'braintree.applepay.product';
    }

    /**
     * Get container id
     *
     * @return string
     */
    public function getContainerId(): string
    {
        return 'braintree-apple-pay-product';
    }

    /**
     * Get product final amount
     *
     * @return float
     * @throws NoSuchEntityException
     */
    public function getAmount(): float
    {
        /** @var Product $product */
        $product = $this->catalogHelper->getProduct();

        if ($product->getTypeId() === Configurable::TYPE_CODE) {
            return (float) $product->getFinalPrice();
        }
        if ($product->getTypeId() === Grouped::TYPE_CODE) {
            $groupedProducts = $product->getTypeInstance()->getAssociatedProducts($product);
            return (float) $groupedProducts[0]->getPrice();
        }

        return (float) $product->getPriceInfo()->getPrice('final_price')->getAmount()->getValue();
    }

    /**
     * Get currency code
     *
     * @return string|null
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function getCurrencyCode(): ?string
    {
        return $this->_storeManager->getStore()->getCurrentCurrency()->getCode();
    }
}
