<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace PayPal\Braintree\Controller\GooglePay;

use Exception;
use Magento\Checkout\Model\Session;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Action\HttpGetActionInterface;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Serialize\Serializer\Json;
use PayPal\Braintree\Model\GooglePay\Config;
use PayPal\Braintree\Model\GooglePay\Helper\QuoteUpdater;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\View\Result\Page;

class Review extends AbstractAction implements HttpGetActionInterface, HttpPostActionInterface
{
    /**
     * @var QuoteUpdater
     */
    private QuoteUpdater $quoteUpdater;

    /**
     * @var string
     */
    private static string $paymentMethodNonce = 'payment_method_nonce';

    /**
     * @var Json
     */
    protected Json $json;

    /**
     * Constructor
     *
     * @param Context $context
     * @param Config $config
     * @param Session $checkoutSession
     * @param QuoteUpdater $quoteUpdater
     * @param Json $json
     */
    public function __construct(
        Context $context,
        Config $config,
        Session $checkoutSession,
        QuoteUpdater $quoteUpdater,
        Json $json
    ) {
        parent::__construct($context, $config, $checkoutSession);
        $this->json = $json;
        $this->quoteUpdater = $quoteUpdater;
    }

    /**
     * @inheritdoc
     */
    public function execute(): Page|Redirect
    {
        $requestData = $this->json->unserialize(
            $this->getRequest()->getPostValue('result', '{}')
        );

        try {
            $quote = $this->checkoutSession->getQuote();
            $this->validateQuote($quote);

            if ($this->validateRequestData($requestData)) {
                $this->quoteUpdater->execute(
                    $requestData['nonce'],
                    $requestData['deviceData'] ?? '',
                    $requestData['details'],
                    $quote
                );
            } elseif (!$quote->getPayment()->getAdditionalInformation(self::$paymentMethodNonce)) {
                throw new LocalizedException(__("We can't initialize checkout."));
            }

            /** @var Page $resultPage */
            $resultPage = $this->resultFactory->create(ResultFactory::TYPE_PAGE);

            /** @var \PayPal\Braintree\Block\GooglePay\Checkout\Review $reviewBlock */
            $reviewBlock = $resultPage->getLayout()->getBlock('braintree.googlepay.review');

            $reviewBlock->setQuote($quote);
            $reviewBlock->getChildBlock('shipping_method')->setData('quote', $quote);

            return $resultPage;
        } catch (Exception $e) {
            $this->messageManager->addExceptionMessage($e, $e->getMessage());
        }

        /** @var Redirect $resultRedirect */
        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);

        return $resultRedirect->setPath('checkout/cart', ['_secure' => true]);
    }

    /**
     * Validate request data
     *
     * @param array $requestData
     * @return bool
     */
    private function validateRequestData(array $requestData): bool
    {
        return !empty($requestData['nonce']) && !empty($requestData['details']);
    }
}
