<?php

declare(strict_types=1);

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Meta\Catalog\Model\Product\Feed\Method;

use GuzzleHttp\Exception\GuzzleException;
use Meta\BusinessExtension\Helper\FBEHelper;
use Meta\BusinessExtension\Helper\GraphAPIAdapter;
use Meta\BusinessExtension\Model\System\Config as SystemConfig;
use Meta\Catalog\Model\Config\Source\FeedUploadMethod;
use Meta\Catalog\Model\Product\Feed\Builder;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Filesystem;
use Magento\Framework\Filesystem\Driver\File;

/**
 * Class Use for Feed Api
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class FeedApi
{
    private const FEED_FILE_NAME = 'facebook_products%s.csv.gz';
    private const FEED_FILE_LOCK_EXT = '.lock';
    private const FB_FEED_NAME = 'Magento Product Feed';

    private const OLD_FB_FEED_NAME = 'Magento Autogenerated Feed';

    /**
     * @var int
     */
    private $storeId;

    /**
     * @var SystemConfig
     */
    private $systemConfig;

    /**
     * @var GraphAPIAdapter
     */
    private $graphApiAdapter;

    /**
     * @var Filesystem
     */
    private $fileSystem;

    /**
     * @var array
     */
    private $productRetrievers;

    /**
     * @var FBEHelper
     */
    private FBEHelper $fbeHelper;

    /**
     * @var Builder
     */
    private $builder;

    /**
     * @var File
     */
    private File $fileHandler;

    private const MAX_PRODUCT_ERRORS_LOGGED = 10;

    /**
     * @param SystemConfig $systemConfig
     * @param GraphAPIAdapter $graphApiAdapter
     * @param Filesystem $filesystem
     * @param File $fileHandler
     * @param array $productRetrievers
     * @param Builder $builder
     * @param FBEHelper $fbeHelper
     */
    public function __construct(
        SystemConfig    $systemConfig,
        GraphAPIAdapter $graphApiAdapter,
        Filesystem      $filesystem,
        File            $fileHandler,
        array           $productRetrievers,
        Builder         $builder,
        FBEHelper       $fbeHelper
    ) {
        $this->systemConfig = $systemConfig;
        $this->graphApiAdapter = $graphApiAdapter;
        $this->fileSystem = $filesystem;
        $this->productRetrievers = $productRetrievers;
        $this->builder = $builder;
        $this->builder->setUploadMethod(FeedUploadMethod::UPLOAD_METHOD_FEED_API);
        $this->fbeHelper = $fbeHelper;
        $this->fileHandler = $fileHandler;
    }

    /**
     * Get FB Feed ID
     *
     * @return mixed|null
     * @throws GuzzleException
     */
    private function getFbFeedId()
    {
        $feedId = $this->systemConfig->getFeedId($this->storeId);
        $feedName = self::FB_FEED_NAME;
        $oldFeedName = self::OLD_FB_FEED_NAME;
        $catalogId = $this->systemConfig->getCatalogId($this->storeId);
        $catalogFeeds = $this->graphApiAdapter->getCatalogFeeds($catalogId);

        // make sure feed exists on meta side, not deleted
        $feedId = $this->verifyFeedExistsInMetaCatalog($feedId, $catalogFeeds);

        // If feedId does not exist in Magento or deleted on Meta,
        // try to find feed with Name
        if (!$feedId) {
            $magentoFeeds = array_filter($catalogFeeds, function ($a) use ($feedName, $oldFeedName) {
                return ($a['name'] === $feedName || $a['name'] === $oldFeedName);
            });
            if (!empty($magentoFeeds)) {
                $feedId = $magentoFeeds[array_key_first($magentoFeeds)]['id'];
            }
            if ($feedId) {
                $this->saveFeedId($feedId);
            }
        }

        // still if feed id is not found, create new feed
        if (!$feedId) {
            $feedId = $this->createNewFeedAndSave($catalogId, $feedName);
        }
        return $feedId;
    }

    /**
     * Creates new feed and save it
     *
     * @param string $catalogId
     * @param string $feedName
     * @return mixed
     * @throws GuzzleException
     */
    private function createNewFeedAndSave(string $catalogId, string $feedName)
    {
        $feedId = $this->graphApiAdapter->createEmptyFeed($catalogId, $feedName);

        $maxAttempts = 5;
        $attempts = 0;
        do {
            $feedData = $this->graphApiAdapter->getFeed($feedId);
            if ($feedData !== false) {
                break;
            }
            $attempts++;
            usleep(2000000);
        } while ($attempts < $maxAttempts);

        if ($feedId) {
            $this->saveFeedId($feedId);
        }

        return $feedId;
    }

    /**
     * Verify is feed exists in Meta Catalog, if not returns null
     *
     * @param string|null $feedId
     * @param array $catalogFeeds
     * @return string|null
     */
    private function verifyFeedExistsInMetaCatalog(?string $feedId, array $catalogFeeds): ?string
    {
        // make sure feed exists on meta side, not deleted
        if ($feedId) {
            $magentoFeeds = array_filter($catalogFeeds, fn($a) => $a['id'] === $feedId);
            // in case feed id is not found in meta catalog, feed id on
            // magento will be flushed and new feed will be created in Meta Catalog
            if (empty($magentoFeeds)) {
                return null;
            }
        }
        return $feedId;
    }

    /**
     * Saves FB Feed ID to configurations
     *
     * @param string $feedId
     * @return void
     */
    private function saveFeedId(string $feedId): void
    {
        $this->systemConfig->saveConfig(
            SystemConfig::XML_PATH_FACEBOOK_BUSINESS_EXTENSION_FEED_ID,
            $feedId,
            $this->storeId
        )->cleanCache();
    }

    /**
     * Write file
     *
     * @param resource $resource
     * @throws \Throwable
     */
    private function writeCsvGzipFile($resource)
    {
        $this->fileHandler->filePutcsv($resource, $this->builder->getHeaderFields());

        $total = 0;
        $totalErrors = 0;
        foreach ($this->productRetrievers as $productRetriever) {
            $productRetriever->setStoreId($this->storeId);
            $offset = 0;
            $limit = $productRetriever->getLimit();
            do {
                $products = $productRetriever->retrieve($offset);
                $offset += $limit;
                if (empty($products)) {
                    break;
                }
                foreach ($products as $product) {
                    try {
                        $entry = array_values($this->builder->buildProductEntry($product));
                        $this->fileHandler->filePutcsv($resource, $entry);
                        $total++;
                    } catch (\Throwable $e) {
                        $totalErrors++;
                        // To prevent all the product errors logs, added cap on number of error logs.
                        if ($totalErrors <= self::MAX_PRODUCT_ERRORS_LOGGED) {
                            $this->fbeHelper->logExceptionImmediatelyToMeta(
                                $e,
                                [
                                    'store_id' => $this->storeId,
                                    'event' => 'feed_upload',
                                    'event_type' => 'feed_upload_product_builder_entry_failed',
                                    'catalog_id' => $this->systemConfig->getCatalogId($this->storeId),
                                    'feed_id' => $this->systemConfig->getFeedId($this->storeId),
                                    'sku' => $product->getSku(),
                                    'product_id' => $product->getId()
                                ]
                            );
                        }
                    }
                }
            } while (true);
        }

        $this->fbeHelper->log(sprintf(
            'Generated feed: Successful Products = %d, Failed Products = %d, StoreID = %d',
            $total,
            $totalErrors,
            $this->storeId
        ));
    }

    /**
     * Get file name with store code suffix for non-default store (no suffix for default one)
     *
     * @return string
     * @throws NoSuchEntityException
     */
    private function getFeedFileName()
    {
        $defaultStoreId = $this->systemConfig->getStoreManager()->getDefaultStoreView()->getId();
        $storeCode = $this->systemConfig->getStoreManager()->getStore($this->storeId)->getCode();
        return sprintf(
            self::FEED_FILE_NAME,
            ($this->storeId && $this->storeId !== $defaultStoreId) ? ('_' . $storeCode) : ''
        );
    }

    /**
     * Generate product feed and push to meta catalog
     *
     * @param string $feedId
     * @return mixed
     * @throws \Throwable
     */
    private function generateAndPushProductFeed(string $feedId)
    {
        $fileName = 'export/' . $this->getFeedFileName();
        $directory = $this->fileSystem->getDirectoryWrite(DirectoryList::VAR_DIR);
        $directory->create('export');
        $filePath = $directory->getAbsolutePath($fileName);

        // Function gzopen does not work along with flock, here we are having another lock file to take lock here.
        $lock = $this->fileHandler->fileOpen($filePath . self::FEED_FILE_LOCK_EXT, "w+");
        // Acquire an exclusive lock on the Gzip file
        $this->fileHandler->fileLock($lock);

        // phpcs:ignore Magento2.Functions.DiscouragedFunction
        $stream = gzopen($filePath, 'w6'); // used compression lvl 6 to have balance b/w CPU and compression data
        $this->writeCsvGzipFile($stream);
        // phpcs:ignore Magento2.Functions.DiscouragedFunction
        gzclose($stream);

        $feedUploadResponse = $this->graphApiAdapter->pushProductFeed($feedId, $filePath);

        // Release the lock from the lock file
        $this->fileHandler->fileUnlock($lock);

        return $feedUploadResponse;
    }

    /**
     * Execute function
     *
     * @param int|null $storeId
     * @throws \Throwable
     */
    public function execute($storeId = null)
    {
        $this->storeId = $storeId;
        $this->builder->setStoreId($this->storeId);
        $accessToken = $this->systemConfig->getAccessToken($storeId);
        if ($accessToken === null) {
            $this->fbeHelper->log(sprintf(
                "Full Catalog Sync: can't find access token, StoreID = %d",
                $storeId
            ));
            return null;
        }
        $this->graphApiAdapter->setDebugMode($this->systemConfig->isDebugMode($storeId))
            ->setAccessToken($accessToken);
        try {
            $feedId = $this->getFbFeedId();
            if (!$feedId) {
                throw new LocalizedException(__('Cannot fetch feed ID'));
            }
            return $this->generateAndPushProductFeed($feedId);
        } catch (\Throwable $e) {
            $this->fbeHelper->logExceptionImmediatelyToMeta(
                $e,
                [
                    'store_id' => $storeId,
                    'event' => 'feed_upload',
                    'event_type' => 'feed_upload',
                    'catalog_id' => $this->systemConfig->getCatalogId($storeId),
                    'feed_id' => $this->systemConfig->getFeedId($storeId)
                ]
            );
            throw $e;
        }
    }
}
