<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\ServicesConnector\Model;

use Magento\ServicesConnector\Api\ClientResolverInterface;
use Magento\ServicesConnector\Api\JwtTokenInterface;
use Magento\ServicesConnector\Exception\KeyNotFoundException;
use Magento\ServicesConnector\Api\KeyValidationInterface;

/**
 * Client resolver implementation
 */
class KeyValidation implements KeyValidationInterface
{
    /**
     * @var EnvironmentFactory
     */
    private $environmentFactory;

    /**
     * @var ClientResolverInterface
     */
    private $clientResolver;

    /**
     * @var JwtTokenInterface $jwtToken
     */
    private $jwtToken;

    /**
     * KeyValidation constructor.
     *
     * @param EnvironmentFactory $environmentFactory
     * @param ClientResolverInterface $clientResolver
     * @param JwtTokenInterface $jwtToken
     */
    public function __construct(
        EnvironmentFactory $environmentFactory,
        ClientResolverInterface $clientResolver,
        JwtTokenInterface $jwtToken
    ) {
        $this->environmentFactory = $environmentFactory;
        $this->clientResolver = $clientResolver;
        $this->jwtToken = $jwtToken;
    }

    /**
     * @inheritDoc
     */
    public function execute($extension, $environment = 'production')
    {
        $envObject = $this->environmentFactory->create($environment);
        if (empty($envObject->getApiKey())) {
            throw new KeyNotFoundException(__("Api key is not found for extension '$extension'"));
        }

        $client = $this->clientResolver->createHttpClient('Magento_ServicesConnector', $environment);
        $privateKey = $envObject->getPrivateKey();

        $url = '/gateway/apikeycheck';
        if (!empty($privateKey)) {
            // Try to sign private key for validation - throws PrivateKeySignException
            $this->jwtToken->getSignature($privateKey);

            $url = $envObject->getKeyValidationUrl();
        }
        if (!$url) {
            // we don't have a URL for this environment
            return true;
        }
        $result = $client->request('GET', $url);
        if ($result->getStatusCode() !== 200) {
            return false;
        }
        return true;
    }
}
