<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\VisualProductRecommendations\Controller\Adminhtml\Index;

use Magento\Backend\App\AbstractAction;
use Magento\Backend\App\Action\Context;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ServicesId\Model\ServicesClientInterface;
use Magento\ServicesId\Model\ServicesConfig;
use Magento\ServicesId\Model\ServicesConfigInterface;
use Psr\Log\LoggerInterface;

/**
 * Controller responsible for communicating with the Magento SaaS Registry service
 */
class EnableVisualRecommendations extends AbstractAction
{
    /**
     * @var ServicesConfigInterface
     */
    private $servicesConfig;

    /**
     * @var ServicesClientInterface
     */
    private $servicesClient;

    /**
     * @var Json
     */
    private $serializer;

    /**
     * @var JsonFactory
     */
    private $resultJsonFactory;

    /**
     * @param Context $context
     * @param ServicesConfigInterface $servicesConfig
     * @param ServicesClientInterface $servicesClient
     * @param Json $serializer
     * @param JsonFactory $resultJsonFactory
     */
    public function __construct(
        Context $context,
        ServicesConfigInterface $servicesConfig,
        ServicesClientInterface $servicesClient,
        Json $serializer,
        JsonFactory $resultJsonFactory
    ) {
        $this->servicesConfig = $servicesConfig;
        $this->servicesClient = $servicesClient;
        $this->serializer = $serializer;
        $this->resultJsonFactory = $resultJsonFactory;
        parent::__construct($context);
    }

    /**
     * Execute middleware call
     *
     * @return ResultInterface
     */
    public function execute() : ResultInterface
    {
        $jsonResult = $this->resultJsonFactory->create();

        $apiVersion = $this->servicesConfig->getRegistryApiVersion();
        $method = $this->getRequest()->getParam('method', 'GET');

        if ($method === 'GET') {
            $uri = 'registry/environments/' . $this->servicesConfig->getEnvironmentId();
            $url = $this->servicesClient->getUrl($apiVersion, $uri);
            $response = $this->servicesClient->request($method, $url);
            $isEnabled = false;
            if (isset($response['featureSet'])) {
                $isEnabled = in_array( 'Visual Recommendations', $response['featureSet']);
            }
            $result = ['visualRecommendationsEnabled' => $isEnabled];
        } else {
            $uri = 'registry/environments/'
                . $this->servicesConfig->getEnvironmentId()
                . '/feature/VISUAL_RECOMMENDATIONS';
            $url = $this->servicesClient->getUrl($apiVersion, $uri);
            $result = $this->servicesClient->request($method, $url);
        }

        return $jsonResult->setData($result);
    }
}
