<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Plugin\Wishlist\Model\ResourceModel\Item;

use Magento\Authorization\Model\UserContextInterface;
use Magento\Company\Model\CompanyContextFactory;
use Magento\Framework\DB\Select;
use Magento\SharedCatalog\Api\SharedCatalogManagementInterface;
use Magento\SharedCatalog\Model\CustomerGroupManagement;
use Magento\SharedCatalog\Model\State;
use Magento\SharedCatalog\Plugin\Wishlist\Model\ResourceModel\Item\ApplyingSharedCatalogFilterToCountSql;
use Magento\Wishlist\Model\ResourceModel\Item\Collection as WishlistCollection;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

class ApplyingSharedCatalogFilterToCountSqlTest extends TestCase
{
    /**
     * @var State|MockObject
     */
    private $sharedCatalogConfig;

    /**
     * @var UserContextInterface|MockObject
     */
    private $userContext;

    /**
     * @var CustomerGroupManagement|MockObject
     */
    private $customerGroupManagement;

    /**
     * @var CompanyContextFactory|MockObject
     */
    private $companyContextFactory;

    /**
     * @var ApplyingSharedCatalogFilterToCountSql
     */
    private $plugin;

    protected function setUp(): void
    {
        $this->sharedCatalogConfig = $this->createMock(State::class);
        $this->userContext = $this->createMock(UserContextInterface::class);
        $this->customerGroupManagement = $this->createMock(CustomerGroupManagement::class);
        $this->companyContextFactory = $this->createMock(CompanyContextFactory::class);

        $this->plugin = new ApplyingSharedCatalogFilterToCountSql(
            $this->sharedCatalogConfig,
            $this->userContext,
            $this->customerGroupManagement,
            $this->companyContextFactory
        );
    }

    /**
     * Test that the shared catalog filter is not applied when the user is an admin.
     *
     * @return void
     */
    public function testAfterGetSelectCountSqlWithAdminUser()
    {
        $wishlistCollection = $this->createMock(WishlistCollection::class);
        $selectMock = $this->createMock(Select::class);
        $this->userContext->method('getUserType')->willReturn(UserContextInterface::USER_TYPE_ADMIN);
        $result = $this->plugin->afterGetSelectCountSql($wishlistCollection, $selectMock);
        $this->assertSame($selectMock, $result);
    }

    /**
     * Test that the shared catalog filter is applied correctly.
     *
     * @return void
     */
    public function testAfterGetSelectCountSqlWithSharedCatalogEnabled()
    {
        $wishlistCollection = $this->createMock(WishlistCollection::class);
        $wishlistCollection->method('getTable')->willReturn('shared_catalog_product_item');

        $selectMock = $this->createMock(Select::class);
        $selectMock->method('getPart')->willReturn(['product_entity' => []]);

        $selectMock->expects($this->once())
            ->method('joinInner')
            ->with(
                ['shared_catalog_item' => 'shared_catalog_product_item'],
                'shared_catalog_item.sku = product_entity.sku',
                []
            )
            ->willReturnSelf();

        $selectMock->expects($this->once())
            ->method('where')
            ->with('shared_catalog_item.customer_group_id = ?')
            ->willReturnSelf();

        $this->sharedCatalogConfig->method('isEnabled')->willReturn(true);
        $this->customerGroupManagement->method('isPrimaryCatalogAvailable')->willReturn(false);

        $companyContext = $this->createMock(\Magento\Company\Model\CompanyContext::class);
        $companyContext->method('getCustomerGroupId')->willReturn(1);
        $this->companyContextFactory->method('create')->willReturn($companyContext);

        $result = $this->plugin->afterGetSelectCountSql($wishlistCollection, $selectMock);
        $this->assertSame($selectMock, $result);
    }
}
