<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Plugin\Catalog\Model\ResourceModel\Product\Indexer;

use Magento\Catalog\Model\ResourceModel\Product\Indexer\LinkedProductSelectBuilderByIndexPrice;
use Magento\Company\Model\CompanyContext;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\LocalizedException;
use \Magento\SharedCatalog\Model\Config;
use Magento\Company\Model\CompanyContextFactory;
use Magento\SharedCatalog\Model\CustomerGroupManagement;
use Magento\SharedCatalog\Plugin\Catalog\Model\ResourceModel\Product\Indexer;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Model\StoreManagerInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class LinkedProductSelectBuilderByIndexPricePluginTest extends TestCase
{
    /**
     * @var MockObject|CompanyContextFactory|CompanyContextFactory&MockObject
     */
    private MockObject|CompanyContextFactory $companyContextFactory;

    /**
     * @var MockObject|StoreManagerInterface|StoreManagerInterface&MockObject
     */
    private MockObject|StoreManagerInterface $storeManager;

    /**
     * @var MockObject|Config|Config&MockObject
     */
    private MockObject|Config $config;

    /**
     * @var MockObject|CustomerGroupManagement|CustomerGroupManagement&MockObject
     */
    private MockObject|CustomerGroupManagement $customerGroupManagement;

    /**
     * @var MockObject|ResourceConnection|ResourceConnection&MockObject
     */
    private MockObject|ResourceConnection $resourceConnection;

    /**
     * @var Indexer\LinkedProductSelectBuilderByIndexPricePlugin
     */
    private Indexer\LinkedProductSelectBuilderByIndexPricePlugin $plugin;

    /**
     * @var MockObject|LinkedProductSelectBuilderByIndexPrice
     */
    private MockObject|LinkedProductSelectBuilderByIndexPrice $subject;

    /**
     * @return void
     */
    protected function setUp(): void
    {
        $this->companyContextFactory = $this->createMock(CompanyContextFactory::class);
        $this->storeManager = $this->createMock(StoreManagerInterface::class);
        $this->config = $this->createMock(Config::class);
        $this->customerGroupManagement = $this->createMock(CustomerGroupManagement::class);
        $this->resourceConnection = $this->createMock(ResourceConnection::class);
        $this->subject = $this->createMock(LinkedProductSelectBuilderByIndexPrice::class);

        $this->plugin = new Indexer\LinkedProductSelectBuilderByIndexPricePlugin(
            $this->companyContextFactory,
            $this->storeManager,
            $this->config,
            $this->customerGroupManagement,
            $this->resourceConnection
        );
    }

    /**
     * @return void
     * @throws LocalizedException
     */
    public function testAfterBuildNoActiveWebsite(): void
    {
        $productId = 5;
        $storeId = 1;
        $websiteId = 1;

        $companyContext = $this->createMock(CompanyContext::class);
        $companyContext->expects($this->once())
            ->method('getCustomerGroupId')
            ->willReturn(1);
        $store = $this->createMock(StoreInterface::class);
        $store->expects($this->once())
            ->method('getWebsiteId')
            ->willReturn($websiteId);
        $this->storeManager->expects($this->once())
            ->method('getStore')
            ->with($storeId)
            ->willReturn($store);
        $this->companyContextFactory->expects($this->once())
            ->method('create')
            ->willReturn($companyContext);

        $this->config->expects($this->once())
            ->method('isActive')
            ->willReturn(false);
        $this->customerGroupManagement->expects($this->never())
            ->method('isPrimaryCatalogAvailable');
        $this->resourceConnection->expects($this->never())
            ->method('getTableName');

        $result = ['test'];
        $this->assertSame($result, $this->plugin->afterBuild($this->subject, $result, $productId, $storeId));
    }

    /**
     * @return void
     * @throws LocalizedException
     */
    public function testAfterBuildPrimaryCatalogNotAvailable(): void
    {
        $productId = 5;
        $storeId = 1;
        $websiteId = 1;

        $companyContext = $this->createMock(CompanyContext::class);
        $companyContext->expects($this->once())
            ->method('getCustomerGroupId')
            ->willReturn(1);
        $store = $this->createMock(StoreInterface::class);
        $store->expects($this->once())
            ->method('getWebsiteId')
            ->willReturn($websiteId);
        $this->storeManager->expects($this->once())
            ->method('getStore')
            ->with($storeId)
            ->willReturn($store);
        $this->companyContextFactory->expects($this->once())
            ->method('create')
            ->willReturn($companyContext);

        $this->config->expects($this->once())
            ->method('isActive')
            ->willReturn(true);
        $this->customerGroupManagement->expects($this->once())
            ->method('isPrimaryCatalogAvailable')
            ->willReturn(true);
        $this->resourceConnection->expects($this->never())
            ->method('getTableName');

        $result = ['test'];
        $this->assertSame($result, $this->plugin->afterBuild($this->subject, $result, $productId, $storeId));
    }

    /**
     * @return void
     * @throws LocalizedException
     */
    public function testAfterBuild(): void
    {
        $productId = 5;
        $storeId = 1;
        $websiteId = 1;

        $companyContext = $this->createMock(CompanyContext::class);
        $companyContext->expects($this->once())
            ->method('getCustomerGroupId')
            ->willReturn(1);
        $store = $this->createMock(StoreInterface::class);
        $store->expects($this->once())
            ->method('getWebsiteId')
            ->willReturn($websiteId);
        $this->storeManager->expects($this->once())
            ->method('getStore')
            ->with($storeId)
            ->willReturn($store);
        $this->companyContextFactory->expects($this->once())
            ->method('create')
            ->willReturn($companyContext);

        $this->config->expects($this->once())
            ->method('isActive')
            ->willReturn(true);
        $this->customerGroupManagement->expects($this->once())
            ->method('isPrimaryCatalogAvailable')
            ->willReturn(false);
        $this->resourceConnection->expects($this->once())
            ->method('getTableName')
            ->willReturn('shared_catalog_product_item');
        $select = $this->createMock(Select::class);
        $select->expects($this->once())
            ->method('joinInner')
            ->willReturnSelf();

        $this->plugin->afterBuild($this->subject, [$select], $productId, $storeId);
    }
}
