<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2017 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Model\Form\Storage;

use Magento\Framework\TestFramework\Unit\Helper\ObjectManager as ObjectManagerHelper;
use Magento\SharedCatalog\Api\CategoryManagementInterface;
use Magento\SharedCatalog\Api\ProductManagementInterface;
use Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Model\Form\Storage\DiffProcessor;
use Magento\SharedCatalog\Model\Form\Storage\Wizard;
use Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price\ScheduleBulk;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for DiffProcessor model.
 */
class DiffProcessorTest extends TestCase
{
    /**
     * @var ObjectManagerHelper
     */
    private $objectManagerHelper;

    /**
     * @var DiffProcessor
     */
    private $diffProcessor;

    /**
     * @var CategoryManagementInterface|MockObject
     */
    private $categoryManagementMock;

    /**
     * @var ProductManagementInterface|MockObject
     */
    private $productManagementMock;

    /**
     * @var ScheduleBulk|MockObject
     */
    private $scheduleBulkMock;

    /**
     * @var SharedCatalogRepositoryInterface|MockObject
     */
    private $sharedCatalogRepositoryMock;

    /**
     * @inheritdoc
     */
    protected function setUp(): void
    {
        $this->categoryManagementMock = $this->getMockBuilder(CategoryManagementInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->productManagementMock = $this->getMockBuilder(ProductManagementInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->scheduleBulkMock = $this->getMockBuilder(ScheduleBulk::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->sharedCatalogRepositoryMock = $this->getMockBuilder(SharedCatalogRepositoryInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->objectManagerHelper = new ObjectManagerHelper($this);
        $this->diffProcessor = $this->objectManagerHelper->getObject(
            DiffProcessor::class,
            [
                'categoryManagement' => $this->categoryManagementMock,
                'productManagement' => $this->productManagementMock,
                'scheduleBulk' => $this->scheduleBulkMock,
                'sharedCatalogRepository' => $this->sharedCatalogRepositoryMock
            ]
        );
    }

    /**
     * @return void
     */
    public function testGetDiff()
    {
        $sharedCatalogId = 1;
        $result = [
            'pricesChanged' => false,
            'categoriesChanged' => false,
            'productsChanged' => false,
            'storeIdChanged' => false
        ];
        $categories = [1];
        $products = ['sku1'];
        $prices = [1];
        $originalStoreId = 1;
        $currentStoreId = 1;

        $storageMock = $this->getMockBuilder(Wizard::class)
            ->disableOriginalConstructor()
            ->getMock();
        $sharedCatalogMock = $this->getMockBuilder(SharedCatalogInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->categoryManagementMock->expects($this->once())->method('getCategories')->with($sharedCatalogId)
            ->willReturn($categories);
        $this->productManagementMock->expects($this->once())->method('getProducts')->with($sharedCatalogId)
            ->willReturn($products);
        $this->sharedCatalogRepositoryMock->expects($this->once())->method('get')->with($sharedCatalogId)
            ->willReturn($sharedCatalogMock);
        $sharedCatalogMock->expects($this->once())->method('getStoreId')->willReturn($originalStoreId);
        $storageMock->expects($this->once())->method('getTierPrices')->willReturn($prices);
        $storageMock->expects($this->once())->method('getUnassignedProductSkus')->willReturn([]);
        $this->scheduleBulkMock->expects($this->once())->method('filterUnchangedPrices')->willReturn([]);
        $storageMock->expects($this->once())->method('getAssignedCategoriesIds')->willReturn($categories);
        $storageMock->expects($this->once())->method('getAssignedProductSkus')->willReturn($products);

        $this->assertEquals($result, $this->diffProcessor->getDiff($storageMock, $sharedCatalogId, $currentStoreId));
    }

    /**
     * Test getDiff with store_id changed.
     *
     * @return void
     */
    public function testGetDiffWithStoreIdChanged()
    {
        $sharedCatalogId = 1;
        $result = [
            'pricesChanged' => false,
            'categoriesChanged' => false,
            'productsChanged' => false,
            'storeIdChanged' => true
        ];
        $categories = [1];
        $products = ['sku1'];
        $prices = [1];
        $originalStoreId = 1;
        $currentStoreId = 2;

        $storageMock = $this->getMockBuilder(Wizard::class)
            ->disableOriginalConstructor()
            ->getMock();
        $sharedCatalogMock = $this->getMockBuilder(SharedCatalogInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->categoryManagementMock->expects($this->once())->method('getCategories')->with($sharedCatalogId)
            ->willReturn($categories);
        $this->productManagementMock->expects($this->once())->method('getProducts')->with($sharedCatalogId)
            ->willReturn($products);
        $this->sharedCatalogRepositoryMock->expects($this->once())->method('get')->with($sharedCatalogId)
            ->willReturn($sharedCatalogMock);
        $sharedCatalogMock->expects($this->once())->method('getStoreId')->willReturn($originalStoreId);
        $storageMock->expects($this->once())->method('getTierPrices')->willReturn($prices);
        $storageMock->expects($this->once())->method('getUnassignedProductSkus')->willReturn([]);
        $this->scheduleBulkMock->expects($this->once())->method('filterUnchangedPrices')->willReturn([]);
        $storageMock->expects($this->once())->method('getAssignedCategoriesIds')->willReturn($categories);
        $storageMock->expects($this->once())->method('getAssignedProductSkus')->willReturn($products);
        $this->assertEquals($result, $this->diffProcessor->getDiff($storageMock, $sharedCatalogId, $currentStoreId));
    }
}
