<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2016 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Block\Adminhtml\SharedCatalog\Edit;

use Magento\Backend\Block\Widget\Context;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Escaper;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager;
use Magento\Framework\UrlInterface;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Block\Adminhtml\SharedCatalog\Edit\DeleteSharedCatalogButton;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Test for Block Adminhtml\SharedCatalog\Edit\DeleteSharedCatalogButton.
 */
class DeleteSharedCatalogButtonTest extends TestCase
{
    /**
     * @var DeleteSharedCatalogButton
     */
    private $deleteSharedCatalogButton;

    /**
     * @var Context|MockObject
     */
    private $context;

    /**
     * @var UrlInterface|MockObject
     */
    private $urlBuilder;

    /**
     * @var RequestInterface|MockObject
     */
    private $request;

    /**
     * @var Escaper|MockObject
     */
    private $escaper;

    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        $this->context = $this->getMockBuilder(Context::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->request = $this->getMockBuilder(RequestInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->urlBuilder = $this->getMockBuilder(UrlInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->escaper = $this->getMockBuilder(Escaper::class)
            ->disableOriginalConstructor()
            ->getMock();

        // Configure context to return our mocked dependencies
        $this->context->expects($this->any())
            ->method('getRequest')
            ->willReturn($this->request);

        $this->context->expects($this->any())
            ->method('getUrlBuilder')
            ->willReturn($this->urlBuilder);

        $this->objectManager = new ObjectManager($this);
        $this->deleteSharedCatalogButton = $this->objectManager->getObject(
            DeleteSharedCatalogButton::class,
            [
                'context' => $this->context,
                'escaper' => $this->escaper
            ]
        );
    }

    /**
     * Test for getButtonData().
     *
     * @return void
     */
    public function testGetButtonData()
    {
        $deleteUrl = 'url/delete';
        $originalMessage = __('This action cannot be undone. Are you sure you want to delete this catalog?');
        $escapedHtml = 'This action cannot be undone. Are you sure you want to delete this catalog?';
        $escapedJs = 'This action cannot be undone. Are you sure you want to delete this catalog?';
        
        $onClickFunction = 'deleteConfirm(\'' . $escapedJs . '\', \'' . $deleteUrl . '\', {data: {}})';
        $expected = [
            'label' => __('Delete'),
            'class' => 'delete',
            'id' => 'shared-catalog-edit-delete-button',
            'on_click' => $onClickFunction,
            'sort_order' => 50,
        ];

        $this->request->expects($this->once())->method('getActionName')->willReturn('edit');

        // Test the nested escaping calls as used in the implementation
        $this->escaper->expects($this->once())
            ->method('escapeHtml')
            ->with($originalMessage)
            ->willReturn($escapedHtml);

        $this->escaper->expects($this->once())
            ->method('escapeJs')
            ->with($escapedHtml)
            ->willReturn($escapedJs);

        $this->prepareGetDeleteUrlMethod($deleteUrl);

        $this->assertEquals($expected, $this->deleteSharedCatalogButton->getButtonData());
    }

    /**
     * Test for getButtonData() when action is not 'edit'.
     *
     * @return void
     */
    public function testGetButtonDataWhenActionIsNotEdit()
    {
        $this->request->expects($this->once())->method('getActionName')->willReturn('index');

        $result = $this->deleteSharedCatalogButton->getButtonData();
        $this->assertEquals([], $result);
    }

    /**
     * Prepare getDeleteUrl().
     *
     * @param string $deleteUrl
     * @return void
     */
    private function prepareGetDeleteUrlMethod($deleteUrl)
    {
        $sharedCatalogParam = SharedCatalogInterface::SHARED_CATALOG_ID_URL_PARAM;
        $sharedCatalogId = 4676;
        $this->request->expects($this->once())->method('getParam')->with($sharedCatalogParam)
            ->willReturn($sharedCatalogId);

        $route = '*/*/delete';
        $routeParams = [$sharedCatalogParam => $sharedCatalogId];
        $this->urlBuilder->expects($this->once())->method('getUrl')->with($route, $routeParams)
            ->willReturn($deleteUrl);
    }

    /**
     * Test for getDeleteUrl().
     *
     * @return void
     */
    public function testGetDeleteUrl()
    {
        $deleteUrl = 'url/delete';
        $this->prepareGetDeleteUrlMethod($deleteUrl);

        $this->assertEquals($deleteUrl, $this->deleteSharedCatalogButton->getDeleteUrl());
    }

    /**
     * Test constructor with null escaper parameter (ObjectManager fallback).
     *
     * @return void
     */
    public function testConstructorWithNullEscaper()
    {
        // Test the ObjectManager fallback behavior
        try {
            $deleteButton = new DeleteSharedCatalogButton(
                $this->context,
                null
            );
            // If we get here, ObjectManager was able to provide an escaper
            $this->assertInstanceOf(DeleteSharedCatalogButton::class, $deleteButton);
        } catch (\RuntimeException $e) {
            // Expected in environments where ObjectManager isn't initialized
            $this->assertStringContainsString("ObjectManager isn't initialized", $e->getMessage());
        }
    }
}
