<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Fixture;

use Magento\Catalog\Api\Data\TierPriceInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\InvalidArgumentException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\SharedCatalog\Api\ProductItemManagementInterface;
use Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface;
use Magento\Store\Model\Store;
use Magento\TestFramework\Fixture\Api\DataMerger;
use Magento\TestFramework\Fixture\Data\ProcessorInterface;
use Magento\TestFramework\Fixture\RevertibleDataFixtureInterface;

class AssignTierPricesToProduct implements RevertibleDataFixtureInterface
{
    private const SHARED_CATALOG_ID = 'shared_catalog_id';
    private const CUSTOMER_GROUP_ID = 'customer_group_id';
    private const PRODUCT_ID = 'product_id';
    private const PRICES = 'prices';
    private const DEFAULT_PRICE_DATA = [
        'qty' => 2,
        'website_id' => 0,
        'value_type' => TierPriceInterface::PRICE_TYPE_FIXED,
        'value' => 8,
        'percentage_value' => null
    ];

    /**
     * @param SharedCatalogRepositoryInterface $sharedCatalogRepository
     * @param ProductItemManagementInterface $productItemManagement
     * @param ProcessorInterface $dataProcessor
     * @param DataMerger $dataMerger
     * @param ProductRepositoryInterface $productRepository
     */
    public function __construct(
        private readonly SharedCatalogRepositoryInterface $sharedCatalogRepository,
        private readonly ProductItemManagementInterface $productItemManagement,
        private readonly ProcessorInterface $dataProcessor,
        private readonly DataMerger $dataMerger,
        private readonly ProductRepositoryInterface $productRepository
    ) {
    }

    /**
     * @inheritdoc
     * @throws InvalidArgumentException|NoSuchEntityException
     */
    public function apply(array $data = []): ?DataObject
    {
        if (empty($data[self::SHARED_CATALOG_ID])) {
            throw new InvalidArgumentException(__('"%field" is required', ['field' => self::SHARED_CATALOG_ID]));
        }
        if (empty($data[self::PRODUCT_ID])) {
            throw new InvalidArgumentException(__('"%field" is required', ['field' => self::PRODUCT_ID]));
        }

        if (empty($data[self::PRICES])) {
            throw new InvalidArgumentException(__('"%field" is required', ['field' => self::PRICES]));
        }

        if (!is_array($data[self::PRICES])) {
            throw new InvalidArgumentException(__('"%field" must be an array', ['field' => self::PRICES]));
        }
        $data[self::PRICES] = $this->prepareData($data[self::PRICES]);
        $sharedCatalog = $this->sharedCatalogRepository->get($data[self::SHARED_CATALOG_ID]);
        $product = $this->productRepository->getById($data[self::PRODUCT_ID], false, Store::DEFAULT_STORE_ID, true);
        $this->productItemManagement->updateTierPrices($sharedCatalog, $product, $data[self::PRICES]);

        return new DataObject($data);
    }

    /**
     * @inheritdoc
     * @param DataObject $data
     * @throws NoSuchEntityException
     */
    public function revert(DataObject $data): void
    {
        $sharedCatalog = $this->sharedCatalogRepository->get($data[self::SHARED_CATALOG_ID]);
        $product = $this->productRepository->getById($data[self::PRODUCT_ID], false, Store::DEFAULT_STORE_ID, true);
        $this->productItemManagement->deleteTierPricesBySku($sharedCatalog, [$product->getSku()]);
    }

    /**
     * Prepare shared catalog tier price data
     *
     * @param array $data
     * @return array
     */
    private function prepareData(array $data): array
    {
        $tierPrices = [];
        foreach ($data as $tierPriceData) {
            $tierPriceData = $this->dataMerger->merge(self::DEFAULT_PRICE_DATA, $tierPriceData);
            $tierPrices[] = $this->dataProcessor->process($this, $tierPriceData);
        }

        return $tierPrices;
    }
}
