<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Fixture;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Exception\InvalidArgumentException;
use Magento\Framework\DataObject;
use Magento\SharedCatalog\Api\ProductManagementInterface;
use Magento\TestFramework\Fixture\RevertibleDataFixtureInterface;

/**
 * Assigning products to shared catalog
 */
class AssignProducts implements RevertibleDataFixtureInterface
{
    private const PRODUCT_IDS = 'product_ids';
    private const CATALOG_ID = 'catalog_id';

    /**
     * @var ProductManagementInterface
     */
    private ProductManagementInterface $productManagement;

    /**
     * @var ProductRepositoryInterface
     */
    private ProductRepositoryInterface $productRepository;

    /**
     * @param ProductManagementInterface $productManagement
     * @param ProductRepositoryInterface $productRepository
     */
    public function __construct(
        ProductManagementInterface $productManagement,
        ProductRepositoryInterface $productRepository
    ) {
        $this->productManagement = $productManagement;
        $this->productRepository = $productRepository;
    }

    /**
     * @inheritdoc
     * @throws InvalidArgumentException
     */
    public function apply(array $data = []): ?DataObject
    {
        if (empty($data[self::CATALOG_ID])) {
            throw new InvalidArgumentException(__('"%field" is required', ['field' => self::CATALOG_ID]));
        }

        if (empty($data[self::PRODUCT_IDS])) {
            throw new InvalidArgumentException(__('"%field" is required', ['field' => self::PRODUCT_IDS]));
        }

        if (!is_array($data[self::PRODUCT_IDS])) {
            throw new InvalidArgumentException(__('"%field" must be an array', ['field' => self::PRODUCT_IDS]));
        }

        $products = [];
        foreach ($data[self::PRODUCT_IDS] as $productId) {
            $products[] = $this->productRepository->getById($productId);
        }
        $data[self::CATALOG_ID] = (int) $data[self::CATALOG_ID];
        $this->productManagement->assignProducts($data[self::CATALOG_ID], $products);

        return new DataObject([
            self::CATALOG_ID => $data[self::CATALOG_ID],
            self::PRODUCT_IDS => $data[self::PRODUCT_IDS],
        ]);
    }

    /**
     * @inheritdoc
     */
    public function revert(DataObject $data): void
    {
        $products = [];
        foreach ($data[self::PRODUCT_IDS] as $productId) {
            $products[] = $this->productRepository->getById($productId);
        }
        $this->productManagement->unassignProducts($data[self::CATALOG_ID], $products);
    }
}
