<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Plugin\Wishlist\Model\ResourceModel\Item;

use Magento\Authorization\Model\UserContextInterface;
use Magento\Company\Model\CompanyContextFactory;
use Magento\SharedCatalog\Model\CustomerGroupManagement;
use Magento\SharedCatalog\Model\State;
use Magento\Wishlist\Model\ResourceModel\Item\Collection as WishlistCollection;
use Magento\Framework\DB\Select;

/**
 * ApplyingSharedCatalogFilterToCountSql
 *
 * This plugin modifies the wishlist item count query to filter by shared catalog.
 */
class ApplyingSharedCatalogFilterToCountSql
{
    /**
     * @var State
     */
    private $sharedCatalogConfig;

    /**
     * @var UserContextInterface
     */
    private $userContext;

    /**
     * @var CustomerGroupManagement
     */
    private $customerGroupManagement;

    /**
     * Customer session factory.
     *
     * @var CompanyContextFactory
     */
    private $companyContextFactory;

    /**
     * @param State $sharedCatalogConfig
     * @param UserContextInterface $userContext
     * @param CustomerGroupManagement $customerGroupManagement
     * @param CompanyContextFactory $companyContextFactory
     *
     */
    public function __construct(
        State $sharedCatalogConfig,
        UserContextInterface $userContext,
        CustomerGroupManagement $customerGroupManagement,
        CompanyContextFactory $companyContextFactory
    ) {
        $this->sharedCatalogConfig = $sharedCatalogConfig;
        $this->userContext = $userContext;
        $this->customerGroupManagement = $customerGroupManagement;
        $this->companyContextFactory = $companyContextFactory;
    }

    /**
     * Modify the select query to include the shared catalog join and filtering by group ID.
     *
     * @param WishlistCollection $subject
     * @param Select $result
     * @return Select
     */
    public function afterGetSelectCountSql(WishlistCollection $subject, Select $result): Select
    {
        // avoid adding shared catalog filter on create/edit products by api
        if ($this->userContext->getUserType() === UserContextInterface::USER_TYPE_ADMIN
            || $this->userContext->getUserType() === UserContextInterface::USER_TYPE_INTEGRATION) {
            return $result;
        }

        $fromTables = $result->getPart(Select::FROM);
        $companyContext = $this->companyContextFactory->create();
        $customerGroupId = $companyContext->getCustomerGroupId();

        if ($this->sharedCatalogConfig->isEnabled()
            && !isset($fromTables['shared_catalog_item'])
            && isset($fromTables['product_entity'])
            && !$this->customerGroupManagement->isPrimaryCatalogAvailable($customerGroupId)
        ) {

            $result->joinInner(
                ['shared_catalog_item' => $subject->getTable('shared_catalog_product_item')],
                'shared_catalog_item.sku = product_entity.sku',
                []
            );

            // Filter by customer group ID
            $result->where('shared_catalog_item.customer_group_id = ?', $customerGroupId);
        }

        return $result;
    }
}
