<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Plugin\Checkout\Model;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Checkout\Model\Cart;
use Magento\SharedCatalog\Model\Permissions\ProductValidator;

/**
 * Plugin to check if product is available under existing shared catalog mapping.
 */
class CartPlugin
{
    /**
     * @param ProductValidator $productValidator
     * @param ProductRepositoryInterface $productRepository
     */
    public function __construct(
        private readonly ProductValidator $productValidator,
        private readonly ProductRepositoryInterface $productRepository,
    ) {
    }

    /**
     * Check if product available in shared catalog.
     *
     * @param Cart $subject
     * @param mixed $productInfo
     * @param mixed|null $requestInfo
     * @return array
     * @throws LocalizedException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function beforeAddProduct(
        Cart $subject,
        mixed $productInfo,
        mixed $requestInfo = null
    ): array {
        $product = $this->getProduct($productInfo);

        if (!$this->productValidator->validate($product)) {
            throw new LocalizedException(
                __('The product that was requested can\'t be added to the cart. '
                    . 'Verify access rights to the product and try again.')
            );
        }

        return [$productInfo, $requestInfo];
    }

    /**
     * Get product object.
     *
     * @param mixed $productInfo
     * @return ?Product
     * @throws LocalizedException
     */
    private function getProduct(mixed $productInfo): ?Product
    {
        if ($productInfo instanceof Product) {
            $product = $productInfo;
        } elseif (is_int($productInfo) || is_string($productInfo)) {
            try {
                $product = $this->productRepository->getById($productInfo);
            } catch (NoSuchEntityException $e) {
                throw new LocalizedException(
                    __('The product wasn\'t found. Verify the product and try again.'),
                    $e
                );
            }
        } else {
            throw new LocalizedException(
                __('The product wasn\'t found. Verify the product and try again.')
            );
        }

        return $product;
    }
}
