<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2016 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
namespace Magento\SharedCatalog\Observer;

use Magento\Company\Api\StatusServiceInterface;
use Magento\Framework\App\Config\ReinitableConfigInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ScopeInterface as AppScopeInterface;
use Magento\Framework\App\ScopeResolverPool;
use Magento\Framework\Event;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\SharedCatalog\Model\Config as SharedCatalogModuleConfig;
use Magento\SharedCatalog\Model\Permissions\Config as PermissionsConfig;
use Magento\SharedCatalog\Model\Permissions\Synchronizer;
use Magento\Store\Model\ScopeInterface;
use Magento\SharedCatalog\Model\CatalogPermissionManagement;
use Magento\Framework\App\Config\ConfigResource\ConfigInterface as ConfigResource;
use Magento\Store\Model\ScopeTypeNormalizer;

/**
 * Additional actions after saving data to system config.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class UpdateConfig implements ObserverInterface
{
    private const CONFIG_DIRECT_PRODUCTS_PRICE_ASSIGNING = 'btob/website_configuration/direct_products_price_assigning';

    /**
     * @var StatusServiceInterface
     */
    private $companyStatusService;

    /**
     * @var SharedCatalogModuleConfig
     */
    private $sharedCatalogModuleConfig;

    /**
     * @var PermissionsConfig
     */
    private $permissionsConfig;

    /**
     * @var CatalogPermissionManagement
     */
    private $catalogPermissionsManagement;

    /**
     * @var ConfigResource
     */
    private $configResource;

    /**
     * @var ScopeResolverPool
     */
    private $scopeResolverPool;

    /**
     * @var Synchronizer
     */
    private $permissionsSynchronizer;

    /**
     * @var ScopeTypeNormalizer
     */
    private $scopeTypeNormalizer;

    /**
     * @var ReinitableConfigInterface
     */
    private $appConfig;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @param StatusServiceInterface $companyStatusService
     * @param SharedCatalogModuleConfig $sharedCatalogModuleConfig
     * @param PermissionsConfig $permissionsConfig
     * @param CatalogPermissionManagement $catalogPermissionsManagement
     * @param ConfigResource $configResource
     * @param ScopeResolverPool $scopeResolverPool
     * @param Synchronizer $permissionsSynchronizer
     * @param ScopeTypeNormalizer $scopeTypeNormalizer
     * @param ReinitableConfigInterface $appConfig
     * @param ScopeConfigInterface $scopeConfig
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        StatusServiceInterface $companyStatusService,
        SharedCatalogModuleConfig $sharedCatalogModuleConfig,
        PermissionsConfig $permissionsConfig,
        CatalogPermissionManagement $catalogPermissionsManagement,
        ConfigResource $configResource,
        ScopeResolverPool $scopeResolverPool,
        Synchronizer $permissionsSynchronizer,
        ScopeTypeNormalizer $scopeTypeNormalizer,
        ReinitableConfigInterface $appConfig,
        ScopeConfigInterface $scopeConfig
    ) {
        $this->companyStatusService = $companyStatusService;
        $this->sharedCatalogModuleConfig = $sharedCatalogModuleConfig;
        $this->permissionsConfig = $permissionsConfig;
        $this->catalogPermissionsManagement = $catalogPermissionsManagement;
        $this->configResource = $configResource;
        $this->scopeResolverPool = $scopeResolverPool;
        $this->permissionsSynchronizer = $permissionsSynchronizer;
        $this->scopeTypeNormalizer = $scopeTypeNormalizer;
        $this->appConfig = $appConfig;
        $this->scopeConfig = $scopeConfig;
    }

    /**
     * Update permissions after updated config values.
     *
     * @param Observer $observer
     * @return void
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function execute(Observer $observer)
    {
        $event = $observer->getEvent();
        $scope = $this->resolveScope($event);

        $changedPaths = (array) $event->getData('changed_paths');
        $isCompanyChanged = \in_array('btob/website_configuration/company_active', $changedPaths);
        $isSharedCatalogChanged = \in_array(SharedCatalogModuleConfig::CONFIG_SHARED_CATALOG, $changedPaths);
        $reInitAppConfig = false;

        $isCompanyActive = $this->companyStatusService->isActive(
            $scope->getScopeType(),
            $scope->getCode()
        );
        $isSharedCatalogActive = $this->sharedCatalogModuleConfig->isActive(
            $scope->getScopeType(),
            $scope->getCode()
        );
        $isDirectProductsPriceAssigningActive = $this->scopeConfig->isSetFlag(
            self::CONFIG_DIRECT_PRODUCTS_PRICE_ASSIGNING,
            $scope->getScopeType(),
            $scope->getCode()
        );

        if ($isCompanyChanged && !$isCompanyActive && $isSharedCatalogActive) {
            $this->configResource->saveConfig(
                SharedCatalogModuleConfig::CONFIG_SHARED_CATALOG,
                0,
                $this->scopeTypeNormalizer->normalize($scope->getScopeType()),
                $scope->getId()
            );

            $isSharedCatalogChanged = 1;
            $isSharedCatalogActive = 0;
            $reInitAppConfig = true;
        }

        if (!$isSharedCatalogActive && $isDirectProductsPriceAssigningActive) {
            $this->configResource->saveConfig(
                self::CONFIG_DIRECT_PRODUCTS_PRICE_ASSIGNING,
                0,
                $this->scopeTypeNormalizer->normalize($scope->getScopeType()),
                $scope->getId()
            );
            $reInitAppConfig = true;
        }

        if ($reInitAppConfig) {
            $this->appConfig->reinit();
        }
        if ($isSharedCatalogChanged) {
            $scopeId = $scope->getId() ? (int) $scope->getId() : null;
            if ($isSharedCatalogActive) {
                $this->permissionsConfig->enable($scopeId);
                $this->catalogPermissionsManagement->setPermissionsForAllCategories($scopeId);
            } else {
                $this->permissionsSynchronizer->removeCategoryPermissions($scopeId);
            }
        }
    }

    /**
     * Resolve scope from event.
     *
     * @param Event $event
     * @return AppScopeInterface
     */
    private function resolveScope(Event $event): AppScopeInterface
    {
        $scopeIdentifier = $event->getWebsite();
        $scopeType = $scopeIdentifier
            ? ScopeInterface::SCOPE_WEBSITES
            : ScopeConfigInterface::SCOPE_TYPE_DEFAULT;
        $scopeResolver = $this->scopeResolverPool->get($scopeType);
        $scope = $scopeResolver->getScope($scopeIdentifier);

        return $scope;
    }
}
