<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2016 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */

namespace Magento\SharedCatalog\Model;

use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\StateException;
use Magento\SharedCatalog\Api\Data\SearchResultsInterfaceFactory;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Api\ProductItemManagementInterface;
use Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface;
use Magento\SharedCatalog\Model\ResourceModel\SharedCatalog;
use Magento\SharedCatalog\Model\ResourceModel\SharedCatalog\CollectionFactory;

/**
 * Shared catalog repository.
 *
 */
class Repository implements SharedCatalogRepositoryInterface
{
    /**
     * List of shared Catalogs.
     *
     * @var SharedCatalogInterface[]
     */
    private $instances = [];

    /**
     * @var SharedCatalog
     */
    private $sharedCatalogResource;

    /**
     * @var CollectionFactory
     */
    private $sharedCatalogCollectionFactory;

    /**
     * @var SearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var ProductItemManagementInterface
     */
    private $sharedCatalogProductItemManagement;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var SharedCatalogValidator
     */
    private $validator;

    /**
     * @var SaveHandler
     */
    private $saveHandler;

    /**
     * Repository constructor.
     *
     * @param SharedCatalog $sharedCatalogResource
     * @param CollectionFactory $sharedCatalogCollectionFactory
     * @param SearchResultsInterfaceFactory $searchResultsFactory
     * @param ProductItemManagementInterface $sharedCatalogProductItemManagement
     * @param CollectionProcessorInterface $collectionProcessor
     * @param SharedCatalogValidator $validator
     * @param SaveHandler $saveHandler
     */
    public function __construct(
        SharedCatalog                  $sharedCatalogResource,
        CollectionFactory              $sharedCatalogCollectionFactory,
        SearchResultsInterfaceFactory  $searchResultsFactory,
        ProductItemManagementInterface $sharedCatalogProductItemManagement,
        CollectionProcessorInterface   $collectionProcessor,
        SharedCatalogValidator         $validator,
        SaveHandler                    $saveHandler
    ) {
        $this->sharedCatalogResource = $sharedCatalogResource;
        $this->sharedCatalogCollectionFactory = $sharedCatalogCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->sharedCatalogProductItemManagement = $sharedCatalogProductItemManagement;
        $this->collectionProcessor = $collectionProcessor;
        $this->validator = $validator;
        $this->saveHandler = $saveHandler;
    }

    /**
     * @inheritdoc
     */
    public function save(SharedCatalogInterface $sharedCatalog)
    {
        if ($sharedCatalog->getId()) {
            $prevSharedCatalogData = $sharedCatalog->getData();
            $sharedCatalog = $this->get($sharedCatalog->getId());
            $sharedCatalog->setData(array_merge($sharedCatalog->getData(), $prevSharedCatalogData));
        }
        $sharedCatalog = $this->saveHandler->execute($sharedCatalog);
        unset($this->instances[$sharedCatalog->getId()]);
        return $sharedCatalog->getId();
    }

    /**
     * @inheritdoc
     */
    public function get($sharedCatalogId)
    {
        if (!isset($this->instances[$sharedCatalogId])) {
            /** @var \Magento\SharedCatalog\Model\ResourceModel\SharedCatalog\Collection $collection */
            $collection = $this->sharedCatalogCollectionFactory->create();
            $collection->addFieldToFilter('entity_id', ['eq' => $sharedCatalogId]);
            /** @var SharedCatalogInterface $sharedCatalog */
            $sharedCatalog = $collection->getFirstItem();
            $this->validator->checkSharedCatalogExist($sharedCatalog);
            $this->instances[$sharedCatalogId] = $sharedCatalog;
        }
        return $this->instances[$sharedCatalogId];
    }

    /**
     * @inheritdoc
     */
    public function delete(SharedCatalogInterface $sharedCatalog)
    {
        if ($this->validator->isSharedCatalogPublic($sharedCatalog)) {
            try {
                $sharedCatalogId = $sharedCatalog->getId();
                $this->sharedCatalogResource->delete($sharedCatalog);
                $this->sharedCatalogProductItemManagement->deleteItems($sharedCatalog);
                unset($this->instances[$sharedCatalogId]);
            } catch (\Exception $e) {
                throw new StateException(
                    __(
                        'Cannot delete shared catalog with id %1',
                        $sharedCatalog->getId()
                    ),
                    $e
                );
            }
        };
        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteById($sharedCatalogId)
    {
        $sharedCatalog = $this->get($sharedCatalogId);
        $this->delete($sharedCatalog);
        return true;
    }

    /**
     * @inheritdoc
     */
    public function getList(\Magento\Framework\Api\SearchCriteriaInterface $criteria)
    {
        /* @var \Magento\SharedCatalog\Api\Data\SearchResultsInterface $searchResults */
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);

        /** @var \Magento\SharedCatalog\Model\ResourceModel\SharedCatalog\Collection $collection */
        $collection = $this->sharedCatalogCollectionFactory->create();
        $this->collectionProcessor->process($criteria, $collection);
        $searchResults->setTotalCount($collection->getSize());
        $searchResults->setItems($collection->getItems());
        return $searchResults;
    }
}
