<?php
/*******************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 ******************************************************************************/
declare(strict_types=1);

namespace Magento\SharedCatalog\Model\Permissions;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Company\Api\CompanyRepositoryInterface;
use Magento\Company\Model\CompanyContextInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\SharedCatalog\Api\Data\ProductItemInterface;
use Magento\SharedCatalog\Api\StatusInfoInterface;
use Magento\SharedCatalog\Model\SharedCatalogResolver;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\SharedCatalog\Model\ResourceModel\ProductItem\CollectionFactory;

/**
 * Validate if product is available under existing shared catalog mapping.
 */
class ProductValidator
{
    /**
     * @param StatusInfoInterface $config
     * @param StoreManagerInterface $storeManager
     * @param SharedCatalogResolver $sharedCatalogResolver
     * @param CompanyRepositoryInterface $companyRepository
     * @param CompanyContextInterface $companyContext
     * @param CollectionFactory $sharedCatalogProductCollectionFactory
     */
    public function __construct(
        private readonly StatusInfoInterface $config,
        private readonly StoreManagerInterface $storeManager,
        private readonly SharedCatalogResolver $sharedCatalogResolver,
        private readonly CompanyRepositoryInterface $companyRepository,
        private readonly CompanyContextInterface $companyContext,
        private readonly CollectionFactory $sharedCatalogProductCollectionFactory,
    ) {
    }

    /**
     * Check if product is available to current group.
     *
     * @param string $sku
     * @param mixed $customerGroupId
     * @return bool
     */
    private function isProductAvailableToGroup(string $sku, int|string $customerGroupId): bool
    {
        $collection = $this->sharedCatalogProductCollectionFactory->create();
        $collection
            ->addFieldToSelect(ProductItemInterface::SKU)
            ->addFieldToFilter(ProductItemInterface::CUSTOMER_GROUP_ID, $customerGroupId)
            ->setPageSize(1);
        $collection->getSelect()->where(\sprintf('%s = ?', ProductItemInterface::SKU), $sku);

        return $collection->getSize() > 0;
    }

    /**
     * Check if product available in shared catalog.
     *
     * @param ProductInterface $product
     * @return bool
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function validate(ProductInterface $product): bool
    {
        $isCompanyConfigEnabled = $this->config->isActive(
            ScopeInterface::SCOPE_WEBSITE,
            $this->storeManager->getWebsite()->getId()
        );
        $companyId = $this->companyContext->getCompanyId();
        if (!$isCompanyConfigEnabled || !$companyId) {
            return true;
        }

        $groupId = (int) $this->companyRepository->get($companyId)->getCustomerGroupId();
        $isPrimaryCatalogAvailable = $this->sharedCatalogResolver->isPrimaryCatalogAvailable($groupId);
        if ($isPrimaryCatalogAvailable) {
            return true;
        }

        return $this->isProductAvailableToGroup($product->getSku(), $groupId);
    }
}
