<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2017 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
namespace Magento\SharedCatalog\Model\Form\Storage;

/**
 * Process changes between shared catalog and wizard storage.
 */
class DiffProcessor
{
    /**
     * @var \Magento\SharedCatalog\Api\CategoryManagementInterface
     */
    private $categoryManagement;

    /**
     * @var \Magento\SharedCatalog\Api\ProductManagementInterface
     */
    private $productManagement;

    /**
     * @var \Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price\ScheduleBulk
     */
    private $scheduleBulk;

    /**
     * @var \Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface
     */
    private $sharedCatalogRepository;

    /**
     * @param \Magento\SharedCatalog\Api\CategoryManagementInterface $categoryManagement
     * @param \Magento\SharedCatalog\Api\ProductManagementInterface $productManagement
     * @param \Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price\ScheduleBulk $scheduleBulk
     * @param \Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface $sharedCatalogRepository
     */
    public function __construct(
        \Magento\SharedCatalog\Api\CategoryManagementInterface $categoryManagement,
        \Magento\SharedCatalog\Api\ProductManagementInterface $productManagement,
        \Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price\ScheduleBulk $scheduleBulk,
        \Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface $sharedCatalogRepository
    ) {
        $this->categoryManagement = $categoryManagement;
        $this->productManagement = $productManagement;
        $this->scheduleBulk = $scheduleBulk;
        $this->sharedCatalogRepository = $sharedCatalogRepository;
    }

    /**
     * Get information whether categories, products, prices or store_id were changed.
     *
     * @param \Magento\SharedCatalog\Model\Form\Storage\Wizard $storage
     * @param int $sharedCatalogId
     * @param int $requestStoreId
     * @return array
     */
    public function getDiff(
        \Magento\SharedCatalog\Model\Form\Storage\Wizard $storage,
        int $sharedCatalogId,
        int $requestStoreId
    ) {
        $origAssignedCategories = $this->categoryManagement->getCategories($sharedCatalogId);
        $origAssignedProducts = $this->productManagement->getProducts($sharedCatalogId);
        $prices = $storage->getTierPrices(null, true);
        $unassignProductSkus = $storage->getUnassignedProductSkus();
        $prices = array_diff_key($prices, array_flip($unassignProductSkus));
        $sharedCatalog = $this->sharedCatalogRepository->get($sharedCatalogId);
        $originalStoreId = $sharedCatalog->getStoreId();

        return [
            'pricesChanged' => (bool)count($this->scheduleBulk->filterUnchangedPrices($prices)),
            'categoriesChanged' =>
                $this->categoriesChanged($storage->getAssignedCategoriesIds(), $origAssignedCategories),
            'productsChanged' => $this->productsChanged($storage->getAssignedProductSkus(), $origAssignedProducts),
            'storeIdChanged' => $this->storeIdChanged($requestStoreId, $originalStoreId)
        ];
    }

    /**
     * Check whether categories were changed.
     *
     * @param array $storageCategoriesIds
     * @param array $origAssignedCategories
     * @return bool
     */
    private function categoriesChanged(array $storageCategoriesIds, array $origAssignedCategories)
    {
        return array_diff($origAssignedCategories, $storageCategoriesIds)
        || array_diff($storageCategoriesIds, $origAssignedCategories);
    }

    /**
     * Check whether products were changed.
     *
     * @param array $storageProductsSkus
     * @param array $origAssignedProducts
     * @return bool
     */
    private function productsChanged(array $storageProductsSkus, array $origAssignedProducts)
    {
        return array_diff($origAssignedProducts, $storageProductsSkus)
        || array_diff($storageProductsSkus, $origAssignedProducts);
    }

    /**
     * Check whether store_id was changed.
     *
     * @param int $currentStoreId
     * @param int|null $originalStoreId
     * @return bool
     */
    private function storeIdChanged(int $currentStoreId, ?int $originalStoreId): bool
    {
        return $currentStoreId !== $originalStoreId;
    }
}
