<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2016 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Model;

use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\ResourceModel\Product as ProductResourceModel;
use Magento\Customer\Api\Data\GroupInterface;
use Magento\Framework\Api\SearchCriteria;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\DataObject;
use Magento\SharedCatalog\Api\CategoryManagementInterface;
use Magento\SharedCatalog\Api\Data\ProductItemInterface;
use Magento\SharedCatalog\Api\Data\ProductItemSearchResultsInterface;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Api\ProductItemManagementInterface;
use Magento\SharedCatalog\Api\ProductItemRepositoryInterface;
use Magento\SharedCatalog\Model\CatalogPermissionManagement;
use Magento\SharedCatalog\Model\ProductItemRepository;
use Magento\SharedCatalog\Model\ProductManagement;
use Magento\SharedCatalog\Model\ResourceModel\CategoryProductLink;
use Magento\SharedCatalog\Model\SharedCatalogInvalidation;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Test ProductManagement model.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class ProductManagementTest extends TestCase
{
    /**
     * @var ProductManagement
     */
    private $productManagement;

    /**
     * @var SearchCriteriaBuilder|MockObject
     */
    private $searchCriteriaBuilder;

    /**
     * @var SharedCatalogInvalidation|MockObject
     */
    private $sharedCatalogInvalidation;

    /**
     * @var CategoryManagementInterface|MockObject
     */
    private $sharedCatalogCategoryManagement;

    /**
     * @var SharedCatalogInterface|MockObject
     */
    private $sharedCatalog;

    /**
     * @var SearchCriteria|MockObject
     */
    private $searchCriteria;

    /**
     * @var ProductItemRepositoryInterface
     */
    private $sharedCatalogProductItemRepository;

    /**
     * @var ProductItemInterface|MockObject
     */
    private $sharedCatalogProduct;

    /**
     * @var Product|MockObject
     */
    private $product;

    /**
     * @var ProductItemManagementInterface|MockObject
     */
    private $sharedCatalogProductItemManagement;

    /**
     * @var CatalogPermissionManagement|MockObject
     */
    private $catalogPermissionManagement;

    /**
     * @var ProductResourceModel|MockObject
     */
    private $productResourceModel;

    /**
     * @var CategoryProductLink|MockObject
     */
    private $categoryProductLink;

    protected function setUp(): void
    {
        $this->sharedCatalogInvalidation = $this->createMock(SharedCatalogInvalidation::class);
        $this->sharedCatalogCategoryManagement = $this->createMock(CategoryManagementInterface::class);
        $this->sharedCatalog = $this->createMock(SharedCatalogInterface::class);
        $this->searchCriteriaBuilder = $this->createMock(SearchCriteriaBuilder::class);
        $this->searchCriteria = $this->createMock(SearchCriteria::class);
        $this->sharedCatalogProductItemRepository = $this->createMock(ProductItemRepository::class);
        $this->sharedCatalogProduct = $this->createMock(ProductItemInterface::class);
        $this->product = $this->createMock(Product::class);
        $this->sharedCatalogProductItemManagement = $this->createMock(ProductItemManagementInterface::class);
        $this->catalogPermissionManagement = $this->createMock(CatalogPermissionManagement::class);
        $this->productResourceModel = $this->createMock(ProductResourceModel::class);
        $this->categoryProductLink = $this->createMock(CategoryProductLink::class);

        $this->productManagement = new ProductManagement(
            $this->sharedCatalogProductItemManagement,
            $this->searchCriteriaBuilder,
            $this->sharedCatalogProductItemRepository,
            $this->sharedCatalogInvalidation,
            $this->sharedCatalogCategoryManagement,
            $this->catalogPermissionManagement,
            $this->productResourceModel,
            $this->categoryProductLink,
            2
        );
    }

    /**
     * Prepare SharedCatalogProductItemRepository mock.
     *
     * @return void
     */
    private function prepareSharedCatalogProductItemRepository()
    {
        $sharedCatalogProductSearchResults = $this
            ->getMockBuilder(ProductItemSearchResultsInterface::class)
            ->onlyMethods(['getItems'])
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $sharedCatalogProducts = [$this->sharedCatalogProduct];
        $sharedCatalogProductSearchResults
            ->expects($this->atLeastOnce())->method('getItems')
            ->willReturn($sharedCatalogProducts);
        $sharedCatalogProductSearchResults->expects($this->any())->method('getTotalCount')->willReturn(1);
        $this->sharedCatalogProductItemRepository
            ->expects($this->atLeastOnce())->method('getList')
            ->willReturn($sharedCatalogProductSearchResults);
    }

    /**
     * Test getProducts().
     *
     * @return void
     */
    public function testGetProducts()
    {
        $sharedCatalogId = 234;

        $customerGroupId = 223;
        $this->sharedCatalog->expects($this->once())->method('getCustomerGroupId')->willReturn($customerGroupId);

        $this->sharedCatalogInvalidation->expects($this->once())->method('checkSharedCatalogExist')
            ->willReturn($this->sharedCatalog);

        $this->searchCriteriaBuilder->expects($this->once())->method('addFilter')->willReturnSelf();
        $this->searchCriteriaBuilder->expects($this->once())->method('create')->willReturn($this->searchCriteria);
        $this->searchCriteria->expects($this->once())->method('setCurrentPage')->with(1);

        $sku = 'HSVC347458';
        $this->sharedCatalogProduct->expects($this->once())->method('getSku')->willReturn($sku);

        $this->prepareSharedCatalogProductItemRepository();

        $result = $this->productManagement->getProducts($sharedCatalogId);

        $this->assertEquals([$sku], $result);
    }

    /**
     * Test assignProducts().
     *
     * @return void
     */
    public function testAssignProducts()
    {
        $sharedCatalogId = 234;
        $sharedCatalogCategoryIds = [7];
        $customerGroupId = 223;
        $customerGroupIds = [$customerGroupId, GroupInterface::NOT_LOGGED_IN_ID];
        $sharedCatalogType = SharedCatalogInterface::TYPE_PUBLIC;
        $skus = ['sku1', 'sku2'];
        $ids = [1, 2];
        $product1 = $this->createMock(Product::class);
        $product1->method('getSku')->willReturn('sku1');
        $product2 = $this->createMock(Product::class);
        $product2->method('getId')->willReturn(2);
        $products = [$product1, $product2];

        $this->productResourceModel->expects($this->once())
            ->method('getProductsSku')
            ->with([2])
            ->willReturn([['entity_id' => 2, 'sku' => 'sku2']]);

        $this->productResourceModel->expects($this->once())
            ->method('getProductsIdsBySkus')
            ->with($skus)
            ->willReturn(array_combine($skus, $ids));

        $this->catalogPermissionManagement
            ->expects($this->once())->method('setAllowPermissions')
            ->with([0 => '5', 2 => '9'], $customerGroupIds);

        $this->sharedCatalogInvalidation->expects($this->once())->method('checkSharedCatalogExist')
            ->willReturn($this->sharedCatalog);

        $this->sharedCatalog->expects($this->once())->method('getCustomerGroupId')->willReturn($customerGroupId);
        $this->sharedCatalog->expects($this->once())->method('getId')->willReturn($sharedCatalogId);
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogType);

        $this->sharedCatalogCategoryManagement->expects($this->once())->method('getCategories')
            ->willReturn($sharedCatalogCategoryIds);
        $this->categoryProductLink->expects($this->once())
            ->method('getCategoryIds')
            ->with($ids)
            ->willReturn([5, 7, 9]);

        $this->sharedCatalogProductItemManagement->expects($this->exactly(2))
            ->method('addItems')
            ->willReturnCallback(
                static function () use ($customerGroupId, $skus) {
                    static $i = 0;
                    $argsSequence = [
                        [$customerGroupId, $skus],
                        [GroupInterface::NOT_LOGGED_IN_ID, $skus]
                    ];
                    self::assertArrayHasKey($i, $argsSequence);
                    self::assertEquals($argsSequence[$i], func_get_args());
                    $i++;
                }
            );

        $this->sharedCatalogInvalidation->expects($this->once())
            ->method('reindexCatalogProductPermissions')
            ->with($ids);

        $this->assertTrue($this->productManagement->assignProducts($sharedCatalogId, $products));
    }

    /**
     * Test unassignProducts().
     *
     * @return void
     * @dataProvider productDataProvider
     */
    public function testUnassignProducts(
        array $sku,
    ) {
        $sharedCatalogId = 234;
        $products = [$this->product];
        $skus = [$sku];
        $this->sharedCatalogInvalidation->expects($this->once())->method('checkSharedCatalogExist')
            ->willReturn($this->sharedCatalog);

        $this->sharedCatalogInvalidation->expects($this->once())->method('validateUnassignProducts')
            ->willReturn($skus);

        $customerGroupId = 223;
        $this->sharedCatalog->expects($this->once())->method('getCustomerGroupId')->willReturn($customerGroupId);
        $sharedCatalogType = SharedCatalogInterface::TYPE_PUBLIC;
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogType);

        //delete items
        $this->searchCriteriaBuilder->expects($this->atLeastOnce())->method('addFilter')->willReturnSelf();
        $this->searchCriteriaBuilder->expects($this->atLeastOnce())
            ->method('create')
            ->willReturn($this->searchCriteria);

        $this->sharedCatalogProduct->expects($this->atLeastOnce())->method('getSku')->willReturn($sku);

        $this->prepareSharedCatalogProductItemRepository();
        $sharedCatalogItemIsDeleted = true;
        $this->sharedCatalogProductItemRepository->expects($this->atLeastOnce())->method('deleteItems')
            ->willReturn($sharedCatalogItemIsDeleted);

        $this->sharedCatalogInvalidation->expects($this->atLeastOnce())->method('cleanCacheByTag');
        $this->sharedCatalogInvalidation->expects($this->atLeastOnce())->method('invalidateIndexRegistryItem');

        $this->assertTrue($this->productManagement->unassignProducts($sharedCatalogId, $products));
    }

    /**
     * Test reassignProducts().
     *
     * @return void
     * @dataProvider productDataProvider
     */
    public function testReassignProducts(
        array $sku
    ) {
        $skus = $sku;
        $customerGroupId = 223;
        $this->sharedCatalog->expects($this->once())->method('getCustomerGroupId')->willReturn($customerGroupId);
        $sharedCatalogType = SharedCatalogInterface::TYPE_PUBLIC;
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogType);

        $this->searchCriteriaBuilder->expects($this->exactly(8))->method('addFilter')->willReturnSelf();
        $this->searchCriteriaBuilder->expects($this->exactly(4))
            ->method('create')
            ->willReturn($this->searchCriteria);

        $this->sharedCatalogProduct->expects($this->exactly(4))->method('getSku')->willReturn($sku);

        $this->prepareSharedCatalogProductItemRepository();
        $sharedCatalogItemIsDeleted = true;
        $this->sharedCatalogProductItemRepository->expects($this->exactly(4))->method('deleteItems')
            ->willReturn($sharedCatalogItemIsDeleted);

        $this->sharedCatalogInvalidation->expects($this->atLeastOnce())->method('cleanCacheByTag');
        $this->sharedCatalogInvalidation->expects($this->atLeastOnce())->method('invalidateIndexRegistryItem');

        $this->sharedCatalogProductItemManagement->expects($this->atLeastOnce())->method('addItems')->willReturnSelf();

        $result = $this->productManagement->reassignProducts($this->sharedCatalog, $skus);
        $this->assertEquals($this->productManagement, $result);
    }

    /**
     * @return array
     */
    public static function productDataProvider()
    {
        return [
            'product data' => [
                ['FGJFG4554345', 'SKU2', 'SKU3'],
                [1, 2, 3],
                [
                    1 =>
                        new DataObject(['entity_id' => 1,
                            'customer_group_id' => 1,
                            'sku' => 'FGJFG4554345']),
                    2 =>
                        new DataObject(['entity_id' => 2,
                            'customer_group_id' => 1,
                            'sku' => 'SKU2']),
                    3 =>
                        new DataObject(['entity_id' => 3,
                            'customer_group_id' => 1,
                            'sku' => 'SKU3'])
                ]
            ]
        ];
    }
}
