<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Plugin\Catalog\Controller\Adminhtml\Product;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use \Magento\Catalog\Controller\Adminhtml\Product\Save as ProductSave;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Api\ProductManagementInterface;
use Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface;
use Magento\SharedCatalog\Model\ProductSharedCatalogsLoader;

class Save
{
    /**
     * @var ProductRepositoryInterface
     */
    private ProductRepositoryInterface $productRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private SearchCriteriaBuilder $searchCriteriaBuilder;

    /**
     * @var SharedCatalogRepositoryInterface
     */
    private SharedCatalogRepositoryInterface $sharedCatalogRepository;

    /**
     * @var ProductSharedCatalogsLoader
     */
    private ProductSharedCatalogsLoader $productSharedCatalogsLoader;

    /**
     * @var ProductManagementInterface
     */
    private ProductManagementInterface $productSharedCatalogManagement;

    /**
     * @param ProductRepositoryInterface $productRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SharedCatalogRepositoryInterface $sharedCatalogRepository
     * @param ProductSharedCatalogsLoader $productSharedCatalogsLoader
     * @param ProductManagementInterface $productSharedCatalogManagement
     */
    public function __construct(
        ProductRepositoryInterface $productRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SharedCatalogRepositoryInterface $sharedCatalogRepository,
        ProductSharedCatalogsLoader $productSharedCatalogsLoader,
        ProductManagementInterface $productSharedCatalogManagement
    ) {
        $this->productRepository = $productRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sharedCatalogRepository = $sharedCatalogRepository;
        $this->productSharedCatalogsLoader = $productSharedCatalogsLoader;
        $this->productSharedCatalogManagement = $productSharedCatalogManagement;
    }

    /**
     * Intercepts product save action and performs shared catalog adjustments
     *
     * @param ProductSave $subject
     * @param callable $proceed
     * @return mixed
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function aroundExecute(ProductSave $subject, callable $proceed): mixed
    {
        try {
            $oldProduct = $this->getProductBeforeSave((int)$subject->getRequest()->getParam('id'));
        } catch (\Throwable) {
            return $proceed();
        }
        $returnValue = $proceed();
        $productRequestData = $subject->getRequest()->getParam('product');
        try {
            $product = $this->productRepository->get($productRequestData['sku']);
        } catch (\Throwable) {
            return $returnValue;
        }

        $sharedCatalogIds = $this->getSharedCatalogIds($productRequestData, $oldProduct);

        $assignedSharedCatalogs = $this->productSharedCatalogsLoader->getAssignedSharedCatalogs($product->getSku());
        $assignedSharedCatalogIds = array_keys($assignedSharedCatalogs);

        $forCreate = array_diff($sharedCatalogIds, $assignedSharedCatalogIds);
        foreach ($forCreate as $sharedCatalogId) {
            $this->productSharedCatalogManagement->assignProducts($sharedCatalogId, [$product]);
        }

        $forDelete = array_diff($assignedSharedCatalogIds, $sharedCatalogIds);
        foreach ($forDelete as $sharedCatalogId) {
            $this->productSharedCatalogManagement->unassignProducts($sharedCatalogId, [$product]);
        }

        return $returnValue;
    }

    /**
     * Get product details before save action is processed
     *
     * @param int $productId
     * @return ProductInterface|null
     * @throws NoSuchEntityException
     */
    private function getProductBeforeSave(int $productId): ?ProductInterface
    {
        $oldProduct = null;
        if ($productId) {
            $oldProduct = clone($this->productRepository->getById(
                $productId,
                true,
                null,
                true
            ));
        }

        return $oldProduct;
    }

    /**
     * Prepare list of shared catalog ids.
     *
     * @param array $sharedCatalogsIds
     * @param array $customerGroupIds
     * @return array
     * @throws LocalizedException
     */
    private function prepareSharedCatalogIds(array $sharedCatalogsIds, array $customerGroupIds): array
    {
        if ($customerGroupIds) {
            $this->searchCriteriaBuilder->addFilter(
                SharedCatalogInterface::CUSTOMER_GROUP_ID,
                $customerGroupIds,
                'in'
            );
            $searchCriteria = $this->searchCriteriaBuilder->create();
            $sharedCatalogs = $this->sharedCatalogRepository->getList($searchCriteria)->getItems();
            $sharedCatalogsIds = [];
            foreach ($sharedCatalogs as $sharedCatalog) {
                $sharedCatalogsIds[] = $sharedCatalog->getId();
            }
        }

        return $sharedCatalogsIds;
    }

    /**
     * Retrieve customer group ids list from tier prices data.
     *
     * @param array $tierPricesData
     * @return array
     */
    private function retrieveCustomerGroupIds(array $tierPricesData): array
    {
        $customerGroups = [];

        foreach ($tierPricesData as $tierPrice) {
            if (!isset($tierPrice['delete']) && !empty($tierPrice['cust_group'])) {
                $customerGroups[] = $tierPrice['cust_group'];
            }
        }

        return $customerGroups;
    }

    /**
     * Generate Shared Catalog ids taking into account configured tier prices
     *
     * @param mixed $productRequestData
     * @param ProductInterface|null $oldProduct
     * @return array
     * @throws LocalizedException
     */
    public function getSharedCatalogIds(mixed $productRequestData, ?ProductInterface $oldProduct): array
    {
        $currentCustomerGroupIds = $this->retrieveCustomerGroupIds(
            !empty($productRequestData['tier_price']) ? (array)$productRequestData['tier_price'] : []
        );
        if ($oldProduct) {
            $oldTierPrices = [];
            foreach ($oldProduct->getTierPrices() as $tierPrice) {
                $oldTierPrices [] = [
                    'cust_group' => $tierPrice->getCustomerGroupId()
                ];
            }
            $previousCustomerGroupIds = $this->retrieveCustomerGroupIds($oldTierPrices);
        } else {
            $previousCustomerGroupIds = [];
        }

        $sharedCatalogIds = (
        !empty($productRequestData['shared_catalog']) ?
            (array)$productRequestData['shared_catalog'] :
            []
        );
        if ($addedCustomerGroupIds = array_diff($currentCustomerGroupIds, $previousCustomerGroupIds)) {
            $addedCatalogIds = $this->prepareSharedCatalogIds(
                $sharedCatalogIds,
                $addedCustomerGroupIds
            );

            $sharedCatalogIds = array_unique(array_merge($sharedCatalogIds, $addedCatalogIds));
        }

        return $sharedCatalogIds;
    }
}
