<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Model;

use Magento\Catalog\Api\Data\TierPriceInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\SharedCatalog\Api\AssignTierPriceInterface;
use Magento\SharedCatalog\Api\ProductItemManagementInterface;
use Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface;

/**
 * Manage tier price for shared catalog
 */
class AssignTierPrice implements AssignTierPriceInterface
{
    private const DEFAULT_WEBSITE_ID = 0;

    /**
     * @var SharedCatalogRepositoryInterface
     */
    private SharedCatalogRepositoryInterface $sharedCatalogRepository;

    /**
     * @var TierPriceManagement
     */
    private TierPriceManagement $sharedCatalogTierPriceManagement;

    /**
     * @var PriceRequestValidator
     */
    private PriceRequestValidator $tierPriceValidator;

    /**
     * Constructor
     *
     * @param SharedCatalogRepositoryInterface $sharedCatalogRepository
     * @param TierPriceManagement $sharedCatalogTierPriceManagement
     * @param PriceRequestValidator $validator
     */
    public function __construct(
        SharedCatalogRepositoryInterface $sharedCatalogRepository,
        TierPriceManagement $sharedCatalogTierPriceManagement,
        PriceRequestValidator $validator
    ) {
        $this->sharedCatalogRepository = $sharedCatalogRepository;
        $this->sharedCatalogTierPriceManagement = $sharedCatalogTierPriceManagement;
        $this->tierPriceValidator = $validator;
    }

    /**
     * @inheritdoc
     */
    public function execute(int $sharedCatalogId, array $prices): void
    {
        try {
            $this->tierPriceValidator->validate($prices);
            $sharedCatalog = $this->sharedCatalogRepository->get($sharedCatalogId);
            foreach ($prices as $price) {
                /** @var TierPriceInterface $price */
                $rawPriceData = [];
                if ($price->getPriceType() === TierPriceInterface::PRICE_TYPE_FIXED) {
                    $rawPriceData['value'] = $price->getPrice();
                    $rawPriceData['value_type'] = TierPriceInterface::PRICE_TYPE_FIXED;
                } else {
                    $rawPriceData['percentage_value'] = $price->getPrice();
                    $rawPriceData['value_type'] = TierPriceInterface::PRICE_TYPE_DISCOUNT;
                }
                $rawPriceData['qty'] = ProductItemManagementInterface::DEFAULT_QTY;
                $rawPriceData['website_id'] = self::DEFAULT_WEBSITE_ID;
                $this->sharedCatalogTierPriceManagement->updateProductTierPrices(
                    $sharedCatalog,
                    $price->getSku(),
                    [$rawPriceData]
                );
            }
        } catch (\Exception $e) {
            throw new LocalizedException(
                __(
                    'Price assignment to Shared Catalog failed: %message',
                    ['message' => $e->getMessage()]
                ),
                $e
            );
        }
    }
}
