<?php
/*******************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 ******************************************************************************/
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Plugin\Wishlist\Model\ResourceModel\Item;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Company\Model\CompanyContext;
use Magento\Company\Model\CompanyContextFactory;
use Magento\Framework\DB\Select;
use Magento\Framework\EntityManager\EntityMetadataInterface;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Exception\LocalizedException;
use Magento\SharedCatalog\Model\CustomerGroupManagement;
use Magento\SharedCatalog\Model\State as SharedCatalogState;
use Magento\SharedCatalog\Plugin\Wishlist\Model\ResourceModel\Item\WishlistItemCollectionPlugin;
use Magento\Wishlist\Model\ResourceModel\Item\Collection;
use PHPUnit\Framework\MockObject\Exception;
use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\MockObject;

class WishlistItemCollectionPluginTest extends TestCase
{
    /**
     * @var WishlistItemCollectionPlugin
     */
    private $plugin;

    /**
     * @var SharedCatalogState|MockObject
     */
    private $sharedCatalogStateMock;

    /**
     * @var CompanyContextFactory|MockObject
     */
    private $companyContextFactoryMock;

    /**
     * @var CustomerGroupManagement|MockObject
     */
    private $customerGroupManagementMock;

    /**
     * @var MetadataPool|MockObject
     */
    private $metadataPoolMock;

    /**
     * @var Collection|MockObject
     */
    private $wishlistItemCollectionMock;

    /**
     * @var Select|MockObject
     */
    private $selectMock;

    protected function setUp(): void
    {
        parent::setUp();

        $this->sharedCatalogStateMock = $this->createMock(SharedCatalogState::class);
        $this->companyContextFactoryMock = $this->createMock(CompanyContextFactory::class);
        $this->customerGroupManagementMock = $this->createMock(CustomerGroupManagement::class);
        $this->metadataPoolMock = $this->createMock(MetadataPool::class);
        $this->wishlistItemCollectionMock = $this->createMock(Collection::class);
        $this->selectMock = $this->createMock(Select::class);

        $this->plugin = new WishlistItemCollectionPlugin(
            $this->sharedCatalogStateMock,
            $this->companyContextFactoryMock,
            $this->customerGroupManagementMock,
            $this->metadataPoolMock
        );
    }

    /**
     * Test to verify that the shared catalog filter is applied to the wishlist items grid collection.
     *
     * @throws Exception
     * @throws LocalizedException
     */
    public function testAfterResetSortOrderWithSharedCatalogEnabled()
    {
        $this->sharedCatalogStateMock->expects($this->once())
            ->method('isEnabled')
            ->willReturn(true);

        $companyContextMock = $this->createMock(CompanyContext::class);
        $companyContextMock->expects($this->once())
            ->method('getCustomerGroupId')
            ->willReturn(1);

        $this->companyContextFactoryMock->expects($this->once())
            ->method('create')
            ->willReturn($companyContextMock);

        $this->customerGroupManagementMock->expects($this->once())
            ->method('isPrimaryCatalogAvailable')
            ->with(1)
            ->willReturn(false);

        $entityMetadataMock = $this->createMock(EntityMetadataInterface::class);
        $entityMetadataMock->expects($this->once())
            ->method('getLinkField')
            ->willReturn('entity_id');

        $this->metadataPoolMock->expects($this->once())
            ->method('getMetadata')
            ->with(ProductInterface::class)
            ->willReturn($entityMetadataMock);

        $this->wishlistItemCollectionMock->expects($this->once())
            ->method('getSelect')
            ->willReturn($this->selectMock);

        $this->wishlistItemCollectionMock->expects($this->exactly(2))
            ->method('getTable')
            ->with($this->logicalOr(
                $this->equalTo('catalog_product_entity'),
                $this->equalTo('shared_catalog_product_item')
            ))
        ->willReturnSelf();

        $this->wishlistItemCollectionMock->expects($this->once())
            ->method('setFlag')
            ->with('has_shared_catalog_filter', true);

        $this->selectMock->expects($this->exactly(2))
            ->method('joinInner')
            ->with(
                $this->callback(function ($args) {
                    if (!empty($args['e'])) {
                        return true;
                    }
                    if (!empty($args['shared_product'])) {
                        return true;
                    }
                    return false;
                }),
                $this->anything(),
                $this->anything()
            )
            ->willReturnSelf();

        $result = $this->plugin->afterResetSortOrder(
            $this->wishlistItemCollectionMock,
            $this->wishlistItemCollectionMock
        );

        $this->assertSame($this->wishlistItemCollectionMock, $result);
    }

    /**
     * When the shared catalog is not enabled, test to make sure the shared catalog filter is not applied.
     *
     * @throws LocalizedException
     */
    public function testAfterResetSortOrderWithSharedCatalogDisabled()
    {
        $this->sharedCatalogStateMock->expects($this->once())
            ->method('isEnabled')
            ->willReturn(false);

        $result = $this->plugin->afterResetSortOrder(
            $this->wishlistItemCollectionMock,
            $this->wishlistItemCollectionMock
        );

        $this->assertSame($this->wishlistItemCollectionMock, $result);
    }

    /**
     * Check to see when the customer group should see the primary catalog;
     * if so, the shard catalog filter shouldn't be applied.
     *
     * @throws LocalizedException
     * @throws Exception
     */
    public function testAfterResetSortOrderWithPrimaryCatalogAvailable()
    {
        $customerGroupId = 1;
        $companyContextMock = $this->createMock(CompanyContext::class);
        $companyContextMock->method('getCustomerGroupId')->willReturn($customerGroupId);
        $this->companyContextFactoryMock->method('create')->willReturn($companyContextMock);

        $this->sharedCatalogStateMock->method('isEnabled')->willReturn(true);
        $this->customerGroupManagementMock->method('isPrimaryCatalogAvailable')->willReturn(true);

        $result = $this->plugin->afterResetSortOrder(
            $this->wishlistItemCollectionMock,
            $this->wishlistItemCollectionMock
        );

        $this->assertSame($this->wishlistItemCollectionMock, $result);
        $this->wishlistItemCollectionMock->expects($this->never())->method('setFlag');
    }
}
