<?php
/**
 * Copyright 2024 Adobe
 * All Rights Reserved.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Model\ResourceModel\SharedCatalog;

use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\StateException;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Framework\Model\ResourceModel\Db\ObjectRelationProcessor;
use Magento\Framework\Model\ResourceModel\Db\TransactionManagerInterface;
use Magento\SharedCatalog\Api\CompanyManagementInterface;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Model\CatalogPermissionManagement;
use Magento\SharedCatalog\Model\CustomerGroupManagement;
use Magento\SharedCatalog\Model\ResourceModel\SharedCatalog;
use Magento\SharedCatalog\Model\SharedCatalog as SharedCatalogModel;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

class SharedCatalogTest extends TestCase
{
    /**
     * @var CompanyManagementInterface|MockObject $companyManagement
     */
    private CompanyManagementInterface $companyManagement;

    /**
     * @var CatalogPermissionManagement|MockObject $catalogPermissionManagement
     */
    private CatalogPermissionManagement $catalogPermissionManagement;

    /**
     * @var CustomerGroupManagement|MockObject $customerGroupManagement
     */
    private CustomerGroupManagement $customerGroupManagement;

    /**
     * @var Context
     */
    private Context $context;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        $this->companyManagement = $this->createMock(CompanyManagementInterface::class);
        $this->catalogPermissionManagement = $this->createMock(CatalogPermissionManagement::class);
        $this->customerGroupManagement = $this->createMock(CustomerGroupManagement::class);
        $this->context = $this->createMock(Context::class);
    }

    public function testDeleteSuccess(): void
    {
        $sharedCatalogModel = $this->createMock(SharedCatalogModel::class);
        $sharedCatalogModel->expects($this->atLeastOnce())->method('getId')->willReturn(1);
        $sharedCatalogModel->expects($this->once())->method('getData')->willReturn([]);
        $sharedCatalogModel->expects($this->once())->method('isDeleted')->with(true);
        $sharedCatalogModel->expects($this->once())->method('afterDelete');
        $sharedCatalogModel->expects($this->once())->method('afterDeleteCommit');

        $adapter = $this->createMock(AdapterInterface::class);
        $resourceConnection = $this->createMock(ResourceConnection::class);
        $resourceConnection->expects($this->any())->method('getConnection')->willReturn($adapter);
        $this->context->expects($this->once())->method('getResources')->willReturn($resourceConnection);

        $transactionManager = $this->createMock(TransactionManagerInterface::class);
        $transactionManager->expects($this->once())->method('start')->willReturn($adapter);
        $transactionManager->expects($this->once())->method('commit');
        $this->context->expects($this->once())
            ->method('getTransactionManager')
            ->willReturn($transactionManager);

        $objectRelationProcessor = $this->createMock(ObjectRelationProcessor::class);
        $this->context->expects($this->once())
            ->method('getObjectRelationProcessor')
            ->willReturn($objectRelationProcessor);

        $objectRelationProcessor->expects($this->once())->method('delete');
        $this->customerGroupManagement->expects($this->once())
            ->method('deleteCustomerGroupById')
            ->with($sharedCatalogModel);

        $sharedCatalogResource = new SharedCatalog(
            $this->context,
            $this->companyManagement,
            $this->catalogPermissionManagement,
            'connectionName',
            $this->customerGroupManagement
        );
        $sharedCatalogResource->delete($sharedCatalogModel);
    }

    public function testCustomerGroupManagerThrowsExceptionCatalogPublic(): void
    {
        $sharedCatalogModel = $this->createMock(SharedCatalogModel::class);
        $sharedCatalogModel->expects($this->atLeastOnce())->method('getId')->willReturn(1);
        $sharedCatalogModel->expects($this->once())->method('getData')->willReturn([]);
        $sharedCatalogModel->expects($this->never())->method('isDeleted');
        $sharedCatalogModel->expects($this->never())->method('afterDelete');
        $sharedCatalogModel->expects($this->never())->method('afterDeleteCommit');
        $sharedCatalogModel->expects($this->once())
            ->method('getType')
            ->willReturn(SharedCatalogInterface::TYPE_PUBLIC);

        $adapter = $this->createMock(AdapterInterface::class);
        $resourceConnection = $this->createMock(ResourceConnection::class);
        $resourceConnection->expects($this->any())->method('getConnection')->willReturn($adapter);
        $this->context->expects($this->once())->method('getResources')->willReturn($resourceConnection);

        $transactionManager = $this->createMock(TransactionManagerInterface::class);
        $transactionManager->expects($this->once())->method('start')->willReturn($adapter);
        $transactionManager->expects($this->once())->method('rollBack');
        $this->context->expects($this->once())
            ->method('getTransactionManager')
            ->willReturn($transactionManager);

        $objectRelationProcessor = $this->createMock(ObjectRelationProcessor::class);
        $this->context->expects($this->once())
            ->method('getObjectRelationProcessor')
            ->willReturn($objectRelationProcessor);

        $objectRelationProcessor->expects($this->once())->method('delete');
        $this->customerGroupManagement->expects($this->once())
            ->method('deleteCustomerGroupById')
            ->with($sharedCatalogModel)
            ->willThrowException(new StateException(__('Cannot delete group.')));

        $sharedCatalogResource = new SharedCatalog(
            $this->context,
            $this->companyManagement,
            $this->catalogPermissionManagement,
            'connectionName',
            $this->customerGroupManagement
        );

        $this->expectException(StateException::class);
        $sharedCatalogResource->delete($sharedCatalogModel);
    }
}
