<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Model\ResourceModel;

use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Model\ResourceModel\Product\Collection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product as ProductResource;
use Magento\Framework\DB\Select;
use Magento\Framework\EntityManager\EntityMetadataInterface;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager;
use Magento\SharedCatalog\Model\ResourceModel\CategoryTree;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * Test for Magento/SharedCatalog/Model/ResourceModel/CategoryTree class.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CategoryTreeTest extends TestCase
{
    /**
     * @var CollectionFactory|MockObject
     */
    private $productCollectionFactory;

    /**
     * @var \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory|MockObject
     */
    private $categoryCollectionFactory;

    /**
     * @var CategoryRepositoryInterface|MockObject
     */
    private $categoryRepository;

    /**
     * @var LoggerInterface|MockObject
     */
    private $logger;

    /**
     * @var MetadataPool|MockObject
     */
    private $metadataPool;

    /**
     * @var CategoryTree
     */
    private $model;

    /**
     * @var ProductResource|MockObject
     */
    private $productResource;

    /**
     * Set up.
     *
     * @return void
     */
    protected function setUp(): void
    {
        $this->productCollectionFactory = $this->getMockBuilder(
            CollectionFactory::class
        )
            ->onlyMethods(['create'])
            ->disableOriginalConstructor()
            ->getMock();
        $this->categoryCollectionFactory = $this->getMockBuilder(
            \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory::class
        )
            ->onlyMethods(['create'])
            ->disableOriginalConstructor()
            ->getMock();
        $this->categoryRepository = $this->getMockBuilder(CategoryRepositoryInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->logger = $this->getMockBuilder(LoggerInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->metadataPool = $this->getMockBuilder(
            MetadataPool::class
        )
            ->disableOriginalConstructor()
            ->getMock();
        $this->productResource = $this->getMockBuilder(ProductResource::class)
            ->onlyMethods(['getProductsIdsBySkus'])
            ->disableOriginalConstructor()
            ->getMock();

        $objectManager = new ObjectManager($this);
        $this->model = $objectManager->getObject(
            CategoryTree::class,
            [
                'productCollectionFactory' => $this->productCollectionFactory,
                'categoryCollectionFactory' => $this->categoryCollectionFactory,
                'categoryRepository' => $this->categoryRepository,
                'logger' => $this->logger,
                'metadataPool' => $this->metadataPool,
                'productResource' => $this->productResource
            ]
        );
    }

    /**
     * Test getCategoryProductsCollectionById method.
     *
     * @param int $level
     * @param $counter
     * @param int $collectionCounter
     * @param int|array $categoryIds
     * @return void
     * @dataProvider getCategoryProductsCollectionByIdDataProvider
     */
    public function testGetCategoryProductsCollectionById(
        $level,
        $counter,
        $collectionCounter,
        $categoryIds
    ) {
        $categoryId = 1;
        $productCollection = $this->getMockBuilder(Collection::class)
            ->disableOriginalConstructor()
            ->getMock();
        $categoryCollection = $this->getMockBuilder(\Magento\Catalog\Model\ResourceModel\Category\Collection::class)
            ->disableOriginalConstructor()
            ->getMock();
        $entity = $this->getMockBuilder(EntityMetadataInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $category = $this->getMockBuilder(CategoryInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $select = $this->getMockBuilder(Select::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->productCollectionFactory->expects($this->once())->method('create')->willReturn($productCollection);
        $this->metadataPool->expects($this->once())
            ->method('getMetadata')
            ->with(ProductInterface::class)
            ->willReturn($entity);
        $entity->expects($this->atLeastOnce())->method('getLinkField')->willReturn('entity_id');
        $entity->expects($this->atLeastOnce())->method('getIdentifierField')->willReturn('entity_id');
        $productCollection
            ->expects($this->once())
            ->method('joinField')
            ->with('position', 'catalog_category_product', 'position', 'product_id=entity_id', null, 'left')
            ->willReturnSelf();
        $this->categoryRepository->expects($this->once())->method('get')->with($categoryId)->willReturn($category);
        $category->expects($this->once())->method('getLevel')->willReturn($level);
        $category->expects($counter)->method('getId')->willReturn(1);
        $this->categoryCollectionFactory
            ->expects($this->exactly($collectionCounter))
            ->method('create')
            ->willReturn($categoryCollection);
        $category->expects($this->exactly($collectionCounter))->method('getPath')->willReturn('1/2');
        $categoryCollection
            ->expects($this->exactly($collectionCounter))
            ->method('addPathsFilter')->with(['1/2'])
            ->willReturnSelf();
        $categoryCollection->expects($this->exactly($collectionCounter))->method('getAllIds')->willReturn($categoryIds);
        $productCollection->expects($this->exactly(3))->method('getSelect')->willReturn($select);
        $select
            ->expects($this->once())
            ->method('where')
            ->with('at_position.category_id IN (?)', $categoryIds)
            ->willReturnSelf();
        $select
            ->expects($this->once())
            ->method('reset')
            ->with(Select::COLUMNS)
            ->willReturnSelf();
        $select
            ->expects($this->once())
            ->method('columns')
            ->with(new \Zend_Db_Expr('DISTINCT e.entity_id, e.sku, e.type_id'))
            ->willReturnSelf();
        $this->assertSame($productCollection, $this->model->getCategoryProductsCollectionById($categoryId));
    }

    /**
     * Test getCategoryProductsCollectionById method throws exception.
     *
     * @return void
     */
    public function testGetCategoryProductsCollectionByIdWithException()
    {
        $categoryId = 1;
        $select = $this->getMockBuilder(Select::class)
            ->disableOriginalConstructor()
            ->getMock();
        $productCollection = $this->getMockBuilder(Collection::class)
            ->disableOriginalConstructor()
            ->getMock();
        $entity = $this->getMockBuilder(EntityMetadataInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $exception = new NoSuchEntityException();
        $this->productCollectionFactory->expects($this->once())->method('create')->willReturn($productCollection);
        $this->metadataPool->expects($this->once())
            ->method('getMetadata')
            ->with(ProductInterface::class)
            ->willReturn($entity);
        $entity->expects($this->atLeastOnce())->method('getLinkField')->willReturn('row_id');
        $entity->expects($this->atLeastOnce())->method('getIdentifierField')->willReturn('entity_id');
        $productCollection
            ->expects($this->once())
            ->method('joinField')
            ->with('position', 'catalog_category_product', 'position', 'product_id=entity_id', null, 'left')
            ->willReturnSelf();
        $this->categoryRepository
            ->expects($this->once())
            ->method('get')
            ->with($categoryId)
            ->willThrowException($exception);
        $this->logger->expects($this->once())->method('critical')->with($exception)->willReturnSelf();
        $productCollection->expects($this->exactly(3))->method('getSelect')->willReturn($select);
        $select
            ->expects($this->once())
            ->method('where')
            ->with('at_position.category_id IN (?)', [])
            ->willReturnSelf();
        $select
            ->expects($this->once())
            ->method('reset')
            ->with(Select::COLUMNS)
            ->willReturnSelf();
        $select
            ->expects($this->once())
            ->method('columns')
            ->with(new \Zend_Db_Expr('DISTINCT e.entity_id, e.row_id, e.sku, e.type_id'))
            ->willReturnSelf();

        $this->assertEquals($productCollection, $this->model->getCategoryProductsCollectionById($categoryId));
    }

    /**
     * Test getCategoryCollection method.
     *
     * @return void
     */
    public function testGetCategoryCollection()
    {
        $categoryId = 1;
        $productSkus = ['SKU1', 'SKU2'];
        $productIdsQuery = 'Load product IDs query';
        $category = $this->getMockBuilder(CategoryInterface::class)
            ->disableOriginalConstructor()
            ->onlyMethods(['getLevel', 'getId', 'getPath'])
            ->getMockForAbstractClass();
        $categoryCollection = $this->getMockBuilder(\Magento\Catalog\Model\ResourceModel\Category\Collection::class)
            ->disableOriginalConstructor()
            ->getMock();
        $connection = $this->getMockBuilder(\Magento\Catalog\Model\ResourceModel\Category\Collection::class)
            ->disableOriginalConstructor()
            ->addMethods(['quote'])
            ->getMockForAbstractClass();
        $categoryCollection
            ->expects($this->any())
            ->method('getConnection')
            ->willReturn($connection);
        $categoryCollection
            ->expects($this->exactly(3))
            ->method('getTable')
            ->willReturnOnConsecutiveCalls(
                'catalog_product_entity',
                'catalog_category_product',
                'catalog_product_entity'
            );
        $this->categoryRepository->expects($this->once())->method('get')->with($categoryId)->willReturn($category);
        $this->categoryCollectionFactory->expects($this->once())->method('create')->willReturn($categoryCollection);
        $category->expects($this->once())->method('getPath')->willReturn('1/2');
        $categoryCollection->expects($this->once())->method('addPathsFilter')->with(['1/2'])->willReturnSelf();
        $categoryCollection
            ->expects($this->exactly(2))
            ->method('addAttributeToSelect')
            ->willReturnCallback(
                function ($arg) use ($categoryCollection) {
                    if ($arg == 'name' || $arg == 'is_active') {
                        return $categoryCollection;
                    }
                }
            );

        $categoryCollection->expects($this->once())->method('setLoadProductCount')->with(false)->willReturnSelf();
        $select = $this->mockDbSelect();
        $categoryCollection->expects($this->exactly(1))->method('getSelect')->willReturn($select);
        $select->expects($this->exactly(1))->method('reset')->willReturnSelf();
        $select
            ->expects($this->exactly(1))
            ->method('from')
            ->willReturnCallback(
                function ($arg1, $arg2) use ($select) {
                    if ($arg1 === ['p' => 'catalog_product_entity'] &&
                        $arg2 === 'COUNT(DISTINCT p.product_id)') {
                        return $select;
                    } elseif ($arg1 === ['p' => 'catalog_category_product'] &&
                        $arg2 === 'COUNT(DISTINCT p.product_id)') {
                        return $select;
                    }
                }
            );
        $select
            ->expects($this->exactly(2))
            ->method('where')
            ->willReturnSelf();
        $select->expects($this->atLeastOnce())->method('__toString')->willReturn($productIdsQuery);
        $categoryCollection
            ->expects($this->once())
            ->method('joinTable')
            ->with(
                ['child_products' => 'catalog_category_product'],
                'category_id=entity_id',
                [
                    'selected_count' => new \Zend_Db_Expr(
                        'COUNT(IF(child_products.product_id IN (1),1,NULL))'
                    ),
                    'product_count' => new \Zend_Db_Expr('COUNT(IF(child_products.product_id IS NULL,NULL,1))'),
                    'root_selected_count' => new \Zend_Db_Expr(
                        '(' . $select . ' AND product.entity_id IN (1))'
                    ),
                    'root_product_count' =>  new \Zend_Db_Expr('(' . $select . ')')
                ],
                null,
                'left'
            )
            ->willReturnSelf();
        $categoryCollection->expects($this->once())->method('groupByAttribute')->with('entity_id')->willReturnSelf();
        $this->productResource->expects($this->any())->method('getProductsIdsBySkus')->willReturn([1]);
        $this->assertSame($categoryCollection, $this->model->getCategoryCollection($categoryId, $productSkus));
    }

    /**
     * Test getCategoryCollection method when product SKUs array is empty.
     *
     * @return void
     */
    public function testGetCategoryCollectionWithEmptyProductSkus()
    {
        $categoryId = 1;
        $productSkus = [];
        $category = $this->getMockForAbstractClass(
            CategoryInterface::class,
            [],
            '',
            false,
            false,
            true,
            ['getPath']
        );
        $categoryCollection = $this->getMockBuilder(\Magento\Catalog\Model\ResourceModel\Category\Collection::class)
            ->disableOriginalConstructor()
            ->getMock();
        $categoryCollection
            ->expects($this->exactly(3))
            ->method('getTable')
            ->willReturnCallback(
                function ($arg) {
                    if ($arg == 'catalog_category_product') {
                        return 'catalog_category_product';
                    } elseif ($arg == 'catalog_product_entity') {
                        return 'catalog_product_entity';
                    } elseif ($arg == 'catalog_category_entity') {
                        return 'catalog_category_entity';
                    }
                }
            );
        $this->categoryRepository->expects($this->once())->method('get')->with($categoryId)->willReturn($category);
        $this->categoryCollectionFactory->expects($this->once())->method('create')->willReturn($categoryCollection);
        $category->expects($this->once())->method('getPath')->willReturn('1/2');
        $categoryCollection->expects($this->once())->method('addPathsFilter')->with(['1/2'])->willReturnSelf();
        $categoryCollection
            ->expects($this->exactly(2))
            ->method('addAttributeToSelect')
            ->willReturnCallback(
                function ($arg) use ($categoryCollection) {
                    if ($arg == 'name') {
                        return $categoryCollection;
                    } elseif ($arg == 'is_active') {
                        return $categoryCollection;
                    }
                }
            );
        $categoryCollection->expects($this->once())->method('setLoadProductCount')->with(false)->willReturnSelf();
        $select = $this->mockDbSelect();
        $categoryCollection->expects($this->once())->method('getSelect')->willReturn($select);
        $select->expects($this->once())->method('reset')->willReturnSelf();
        $select
            ->expects($this->exactly(2))
            ->method('where')
            ->willReturnCallback(
                function ($arg1, $arg2) use ($select) {
                    if ($arg1 == 'ce.path = e.path OR ce.path LIKE CONCAT(e.path, "/%")') {
                        return $select;
                    } elseif ($arg1 == 'e.level IN(?)' && $arg2 === [0, 1]) {
                        return $select;
                    }
                }
            );
        $select
            ->expects($this->once())
            ->method('from')
            ->with(
                ['p' => 'catalog_category_product'],
                'COUNT(DISTINCT p.product_id)'
            )
            ->willReturnSelf();
        $categoryCollection
            ->expects($this->once())
            ->method('joinTable')
            ->with(
                ['child_products' => 'catalog_category_product'],
                'category_id=entity_id',
                [
                    'selected_count' => '',
                    'product_count' => new \Zend_Db_Expr('COUNT(IF(child_products.product_id IS NULL,NULL,1))'),
                    'root_selected_count' => '',
                    'root_product_count' =>  new \Zend_Db_Expr('(' . $select . ')')
                ],
                null,
                'left'
            )
            ->willReturnSelf();
        $categoryCollection->expects($this->once())->method('groupByAttribute')->with('entity_id')->willReturnSelf();
        $this->assertSame($categoryCollection, $this->model->getCategoryCollection($categoryId, $productSkus));
    }

    /**
     * Mock Db Select object.
     *
     * @return MockObject
     */
    private function mockDbSelect()
    {
        $select = $this->getMockBuilder(Select::class)
            ->disableOriginalConstructor()
            ->getMock();
        $select
            ->expects($this->exactly(2))
            ->method('joinInner')
            ->willReturnSelf();

        return $select;
    }

    /**
     * Data provider for getCategoryRootNode method.
     *
     * @return array
     */
    public static function getCategoryProductsCollectionByIdDataProvider()
    {
        return [
            [3, self::once(), 0, 1],
            [1, self::never(), 1, [5, 6, 7]]
        ];
    }
}
