<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Model;

use Magento\Catalog\Api\Data\TierPriceInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\SharedCatalog\Api\AssignTierPriceInterface;
use Magento\SharedCatalog\Api\ProductItemManagementInterface;
use Magento\SharedCatalog\Api\SharedCatalogRepositoryInterface;

/**
 * Manage tier price for shared catalog
 */
class AssignTierPrice implements AssignTierPriceInterface
{
    private const DEFAULT_WEBSITE_ID = 0;

    /**
     * @var SharedCatalogRepositoryInterface
     */
    private SharedCatalogRepositoryInterface $sharedCatalogRepository;

    /**
     * @var TierPriceManagement
     */
    private TierPriceManagement $sharedCatalogTierPriceManagement;

    /**
     * @var PriceRequestValidator
     */
    private PriceRequestValidator $tierPriceValidator;

    /**
     * Constructor
     *
     * @param SharedCatalogRepositoryInterface $sharedCatalogRepository
     * @param TierPriceManagement $sharedCatalogTierPriceManagement
     * @param PriceRequestValidator $validator
     */
    public function __construct(
        SharedCatalogRepositoryInterface $sharedCatalogRepository,
        TierPriceManagement $sharedCatalogTierPriceManagement,
        PriceRequestValidator $validator
    ) {
        $this->sharedCatalogRepository = $sharedCatalogRepository;
        $this->sharedCatalogTierPriceManagement = $sharedCatalogTierPriceManagement;
        $this->tierPriceValidator = $validator;
    }

    /**
     * @inheritdoc
     */
    public function execute(int $sharedCatalogId, array $prices): void
    {
        try {
            $this->tierPriceValidator->validate($prices);
            $sharedCatalog = $this->sharedCatalogRepository->get($sharedCatalogId);
            foreach ($prices as $price) {
                /** @var TierPriceInterface $price */
                $rawPriceData = [];
                if ($price->getPriceType() === TierPriceInterface::PRICE_TYPE_FIXED) {
                    $rawPriceData['value'] = $price->getPrice();
                    $rawPriceData['value_type'] = TierPriceInterface::PRICE_TYPE_FIXED;
                } else {
                    $rawPriceData['percentage_value'] = $price->getPrice();
                    $rawPriceData['value_type'] = TierPriceInterface::PRICE_TYPE_DISCOUNT;
                }
                $rawPriceData['qty'] = ProductItemManagementInterface::DEFAULT_QTY;
                $rawPriceData['website_id'] = self::DEFAULT_WEBSITE_ID;
                $this->sharedCatalogTierPriceManagement->updateProductTierPrices(
                    $sharedCatalog,
                    $price->getSku(),
                    [$rawPriceData]
                );
            }
        } catch (\Exception $e) {
            throw new LocalizedException(
                __(
                    'Price assignment to Shared Catalog failed: %message',
                    ['message' => $e->getMessage()]
                ),
                $e
            );
        }
    }
}
