<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Ui\DataProvider\Configure;

use Magento\Catalog\Model\ResourceModel\Product\Collection;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\DataObject;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Model\Form\Storage\Wizard;
use Magento\SharedCatalog\Model\Form\Storage\WizardFactory;
use Magento\SharedCatalog\Model\ResourceModel\CategoryTree;
use Magento\SharedCatalog\Ui\DataProvider\Configure\Pricing;
use Magento\SharedCatalog\Ui\DataProvider\Configure\StepDataProcessor;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\ResourceModel\Product as ProductResource;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Test for pricing data provider.
 */
class PricingTest extends TestCase
{
    /**
     * @var StepDataProcessor|MockObject
     */
    private $stepDataProcessor;

    /**
     * @var Wizard|MockObject
     */
    private $storage;

    /**
     * @var CategoryTree|MockObject
     */
    private $categoryTree;

    /**
     * @var RequestInterface|MockObject
     */
    private $request;

    /**
     * @var StoreManagerInterface|MockObject
     */
    private $storeManager;

    /**
     * @var Pricing
     */
    private $dataProvider;

    /**
     * @var Collection|MockObject
     */
    private $productCollection;

    /**
     * Dummy category ID.
     *
     * @var int
     */
    private $categoryId = 12;

    /**
     * @var ProductResource|MockObject
     */
    private $productResource;

    /**
     * Set up for test.
     *
     * @return void
     */
    protected function setUp(): void
    {
        $this->stepDataProcessor = $this->getMockBuilder(
            StepDataProcessor::class
        )
            ->disableOriginalConstructor()
            ->getMock();
        $this->storage = $this->getMockBuilder(Wizard::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->categoryTree = $this->getMockBuilder(CategoryTree::class)
            ->disableOriginalConstructor()
            ->getMock();
        $wizardStorageFactory = $this
            ->getMockBuilder(WizardFactory::class)
            ->onlyMethods(['create'])
            ->disableOriginalConstructor()
            ->getMock();
        $this->request = $this->getMockBuilder(RequestInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->storeManager = $this->getMockForAbstractClass(StoreManagerInterface::class);
        $this->productResource = $this->getMockBuilder(ProductResource::class)
            ->onlyMethods(['getProductsIdsBySkus'])
            ->disableOriginalConstructor()
            ->getMock();

        $objectManager = new ObjectManager($this);
        $this->dataProvider = $objectManager->getObject(
            Pricing::class,
            [
                'stepDataProcessor' => $this->stepDataProcessor,
                'request' => $this->request,
                'categoryTree' => $this->categoryTree,
                'wizardStorageFactory' => $wizardStorageFactory,
                'storage' => $this->storage,
                'storeManager' => $this->storeManager,
                'productResource' => $this->productResource
            ]
        );
    }

    /**
     * Test for getData() method when request websites filter is empty.
     *
     * @param array $customPrice
     * @param int $price
     * @param string $priceType
     * @param array $itemData
     * @return void
     * @dataProvider getDataDataProvider
     */
    public function testGetDataWithEmptyRequestWebsitesFilter(array $customPrice, $price, $priceType, array $itemData)
    {
        $websiteId = 3;
        $requestParams = [
            'filters' => ['category_id' => $this->categoryId]
        ];

        $this->prepareGetDataMocks($customPrice, $price, $priceType, $itemData);

        $this->request->expects($this->once())->method('getParams')
            ->willReturn($requestParams);
        $this->request->expects($this->atLeastOnce())->method('getParam')
            ->willReturnCallback(function ($arg) use ($requestParams, $websiteId) {
                if ($arg == 'filters') {
                    return $requestParams['filters'];
                } elseif ($arg == SharedCatalogInterface::SHARED_CATALOG_ID_URL_PARAM) {
                    return $websiteId;
                }
            });
        $this->storage->expects($this->once())->method('getAssignedProductSkus')->willReturn(['sku_1']);
        $this->productResource->expects($this->any())->method('getProductsIdsBySkus')->willReturn([1]);
        $this->productCollection->expects($this->once())
            ->method('addAttributeToFilter')
            ->with('entity_id', ['in' => [1]])
            ->willReturnSelf();
        $this->stepDataProcessor->expects($this->once())->method('retrieveSharedCatalogWebsiteIds')
            ->willReturn([$websiteId]);
        $this->productCollection->expects($this->once())->method('addWebsiteFilter')
            ->with([$websiteId])->willReturnSelf();

        $this->stepDataProcessor->expects($this->once())->method('getWebsites')->willReturn([$websiteId]);

        $this->assertSame($itemData, $this->dataProvider->getData());
    }

    /**
     * Test for getData() method.
     *
     * @param array $customPrice
     * @param int $price
     * @param string $priceType
     * @param array $itemData
     * @return void
     * @dataProvider getDataDataProvider
     */
    public function testGetData(array $customPrice, $price, $priceType, array $itemData)
    {
        $websiteId = 3;
        $requestParams = [
            'filters' => [
                'websites' => 3,
                'category_id' => $this->categoryId
            ]
        ];
        $this->prepareGetDataMocks($customPrice, $price, $priceType, $itemData);

        $this->request->expects($this->once())->method('getParams')
            ->willReturn($requestParams);
        $this->request->expects($this->atLeastOnce())->method('getParam')
            ->with('filters')
            ->willReturn($requestParams['filters']);
        $this->storage->expects($this->once())->method('getAssignedProductSkus')->willReturn(['sku_1']);
        $this->productResource->expects($this->any())->method('getProductsIdsBySkus')->willReturn([1]);
        $this->productCollection->expects($this->once())
            ->method('addAttributeToFilter')
            ->with('entity_id', ['in' => [1]])
            ->willReturnSelf();
        $this->productCollection->expects($this->once())->method('addWebsiteFilter')
            ->with($websiteId)->willReturnSelf();

        $this->stepDataProcessor->expects($this->once())->method('getWebsites')->willReturn([$websiteId]);

        $this->assertSame($itemData, $this->dataProvider->getData());
    }

    /**
     * Prepare mocks for testGetData() method.
     *
     * @param array $customPrice
     * @param int $price
     * @param string $priceType
     * @param array $itemData
     * @return void
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    private function prepareGetDataMocks(array $customPrice, $price, $priceType, array $itemData)
    {
        $sku = 'sku_1';
        $this->productCollection = $this->getMockBuilder(Collection::class)
            ->disableOriginalConstructor()
            ->getMock();
        $item = $this->getMockBuilder(DataObject::class)
            ->disableOriginalConstructor()
            ->addMethods(['getSku', 'setCustomPrice', 'setPriceType', 'getPrice'])
            ->onlyMethods(['setData', 'toArray'])
            ->getMock();
        $this->categoryTree->expects($this->once())
            ->method('getCategoryProductsCollectionById')
            ->with($this->categoryId)
            ->willReturn($this->productCollection);
        $this->productCollection->expects($this->once())->method('getSize')->willReturn(1);
        $this->productCollection->expects($this->once())->method('getItems')->willReturn([$item]);
        $this->stepDataProcessor->expects($this->once())->method('switchCurrentStore');
        $this->stepDataProcessor->expects($this->once())
            ->method('prepareCustomPrice')
            ->with($customPrice)
            ->willReturn($customPrice);
        $item->expects($this->atLeastOnce())->method('getSku')->willReturn($sku);
        $item->expects($this->atLeastOnce())->method('getPrice')->willReturn($price);
        $this->storage->expects($this->once())
            ->method('getProductPrices')
            ->with($sku)
            ->willReturn($customPrice);
        $item->expects($this->once())->method('setCustomPrice')->with($price)->willReturnSelf();
        $item->expects($this->once())->method('setPriceType')->with($priceType)->willReturnSelf();
        $this->storage->expects($this->atLeastOnce())
            ->method('getTierPrices')
            ->willReturnCallback(
                function ($arg1, $arg2) use ($sku) {
                    if ($arg1 == $sku && $arg2 == false) {
                        return ['tier_price'];
                    } elseif ($arg1 === null && $arg2 === true) {
                        return [[['is_changed' => true]]];
                    }
                }
            );
        $this->stepDataProcessor->expects($this->once())
            ->method('isCustomPriceEnabled')
            ->with($customPrice)
            ->willReturn(true);
        $item->expects($this->exactly(3))
            ->method('setData')
            ->willReturnCallback(
                function ($arg1, $arg2) use ($price, $item) {
                    if ($arg1 === 'origin_price' && $arg2 === $price) {
                        return $item;
                    } elseif ($arg1 === 'tier_price_count' && $arg2 === 1) {
                        return $item;
                    } elseif ($arg1 === 'custom_price_enabled' && $arg2 === true) {
                        return $item;
                    }
                }
            );
        $item->expects($this->once())->method('toArray')->willReturn($itemData);
        $this->stepDataProcessor->expects($this->once())
            ->method('modifyData')
            ->with(
                [
                    'totalRecords' => 1,
                    'items' => [0 => $itemData]
                ]
            )
            ->willReturn($itemData);
    }

    /**
     * Data provider for getData method.
     *
     * @return array
     */
    public static function getDataDataProvider()
    {
        return [
            [
                ['value_type' => 'fixed', 'price' => 10],
                10,
                'fixed',
                [
                    'custom_price' => 10,
                    'price_type' => 'fixed',
                    'origin_price' => 10,
                    'tier_price_count' => 1,
                    'custom_price_enabled' => true,
                    'websites' => [3],
                    'isChanged' => true
                ]
            ],
            [
                ['value_type' => 'percent', 'percentage_value' => 5],
                5,
                'percent',
                [
                    'custom_price' => 5,
                    'price_type' => 'percent',
                    'origin_price' => 5,
                    'tier_price_count' => 1,
                    'custom_price_enabled' => true,
                    'websites' => [3],
                    'isChanged' => true
                ]
            ],
        ];
    }
}
