<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Model;

use Magento\Framework\DataObject;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager as ObjectManagerHelper;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Api\SharedCatalogManagementInterface;
use Magento\SharedCatalog\Model\ResourceModel\SharedCatalog\CollectionFactory;
use Magento\SharedCatalog\Model\SharedCatalog;
use Magento\SharedCatalog\Model\SharedCatalogValidator;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Api\GroupRepositoryInterface;
use Magento\Store\Api\StoreRepositoryInterface;
use Magento\Tax\Model\ResourceModel\TaxClass\Collection;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Test for SharedCatalogValidator model.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class SharedCatalogValidatorTest extends TestCase
{
    /**
     * @var ObjectManagerHelper
     */
    private $objectManagerHelper;

    /**
     * @var SharedCatalogValidator
     */
    private $validator;

    /**
     * @var SharedCatalogManagementInterface|MockObject
     */
    private $sharedCatalogManagement;

    /**
     * @var CollectionFactory|MockObject
     */
    private $sharedCatalogCollectionFactory;

    /**
     * @var StoreRepositoryInterface|MockObject
     */
    private $storeGroupRepository;

    /**
     * @var \Magento\Tax\Model\ResourceModel\TaxClass\CollectionFactory|MockObject
     */
    private $taxClassCollectionFactory;

    /**
     * @var SharedCatalogInterface|MockObject
     */
    private $sharedCatalog;

    /**
     * Set up.
     *
     * @return void
     */
    protected function setUp(): void
    {
        $this->sharedCatalogManagement = $this
            ->getMockBuilder(SharedCatalogManagementInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->sharedCatalogCollectionFactory = $this
            ->getMockBuilder(CollectionFactory::class)
            ->onlyMethods(['create'])
            ->disableOriginalConstructor()
            ->getMock();
        $this->storeGroupRepository = $this->getMockBuilder(GroupRepositoryInterface::class)
            ->onlyMethods(['get'])
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->taxClassCollectionFactory = $this
            ->getMockBuilder(\Magento\Tax\Model\ResourceModel\TaxClass\CollectionFactory::class)
            ->onlyMethods(['create'])
            ->disableOriginalConstructor()
            ->getMock();
        $this->sharedCatalog = $this->getMockBuilder(SharedCatalogInterface::class)
            ->onlyMethods([
                'getId', 'getName', 'getStoreId', 'getTaxClassId', 'getType',
                'getCustomerGroupId'
            ])
            ->addMethods(['getAvailableTypes'])
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->objectManagerHelper = new ObjectManagerHelper($this);
        $this->validator = $this->objectManagerHelper->getObject(
            SharedCatalogValidator::class,
            [
                'sharedCatalogManagement' => $this->sharedCatalogManagement,
                'sharedCatalogCollectionFactory' => $this->sharedCatalogCollectionFactory,
                'storeGroupRepository' => $this->storeGroupRepository,
                'taxClassCollectionFactory' => $this->taxClassCollectionFactory
            ]
        );
    }

    /**
     * Prepare TaxClassCollectionFactory mock.
     *
     * @param int $taxClassId
     * @return void
     */
    private function prepareTaxClassCollectionFactory($taxClassId)
    {
        $taxClass = $this->getMockBuilder(DataObject::class)
            ->addMethods(['getId'])
            ->disableOriginalConstructor()
            ->getMock();
        $taxClass->expects($this->exactly(2))->method('getId')->willReturn($taxClassId);

        $taxClassCollection = $this->getMockBuilder(Collection::class)
            ->onlyMethods(['getItems'])
            ->disableOriginalConstructor()
            ->getMock();
        $taxClasses = [$taxClass];
        $taxClassCollection->expects($this->once())->method('getItems')->willReturn($taxClasses);

        $this->taxClassCollectionFactory->expects($this->once())->method('create')
            ->willReturn($taxClassCollection);
    }

    /**
     * Prepare SharedCatalogCollectionFactory mock.
     *
     * @param array $calls
     * @return void
     */
    private function prepareSharedCatalogCollectionFactory(array $calls)
    {
        $sharedCatalogCollection = $this
            ->getMockBuilder(\Magento\SharedCatalog\Model\ResourceModel\SharedCatalog\Collection::class)
            ->onlyMethods(['addFieldToFilter', 'getFirstItem'])
            ->disableOriginalConstructor()
            ->getMock();
        $sharedCatalogCollection->expects($this->exactly($calls['sharedCatalogCollection_addFieldToFilter']))
            ->method('addFieldToFilter')->willReturnSelf();

        $sharedCatalogCollection->expects($this->exactly($calls['sharedCatalogCollection_getFirstItem']))
            ->method('getFirstItem')->willReturn($this->sharedCatalog);

        $this->sharedCatalogCollectionFactory->expects($this->exactly($calls['sharedCatalogCollectionFactory_create']))
            ->method('create')->willReturn($sharedCatalogCollection);
    }

    /**
     * Prepare validate().
     *
     * @param array $returned
     * @param array $calls
     * @return void
     */
    private function prepareValidateMethod(array $returned, array $calls)
    {
        $this->prepareTaxClassCollectionFactory($returned['taxClass_getId']);

        $sharedCatalogName = isset($returned['sharedCatalogName']) ?
            $returned['sharedCatalogName'] : 'Test Shared Catalog';
        $this->sharedCatalog->expects($this->exactly($calls['sharedCatalog_getName']))->method('getName')
            ->willReturn($sharedCatalogName);
        $storeId = 4;
        $this->sharedCatalog->expects($this->exactly($calls['sharedCatalog_getStoreId']))->method('getStoreId')
            ->willReturn($storeId);
        $this->sharedCatalog->expects($this->exactly($calls['sharedCatalog_getTaxClassId']))->method('getTaxClassId')
            ->willReturn($returned['sharedCatalog_getTaxClassId']);
        $this->sharedCatalog->expects($this->exactly($calls['sharedCatalog_getType']))->method('getType')
            ->willReturn($returned['sharedCatalog_getType']);
        $this->sharedCatalog->expects($this->exactly($calls['sharedCatalog_getAvailableTypes']))
            ->method('getAvailableTypes')->willReturn($returned['sharedCatalog_getAvailableTypes']);

        $store = $this->getMockBuilder(StoreInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->storeGroupRepository->expects($this->exactly($calls['storeRepository_getById']))->method('get')
            ->willReturn($store);

        $this->prepareSharedCatalogCollectionFactory($calls);
    }

    /**
     * Test for validate().
     *
     * @return void
     */
    public function testValidate()
    {
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 4, 'sharedCatalogCollection_getFirstItem' => 3,
            'sharedCatalogCollectionFactory_create' => 3, 'sharedCatalog_getName' => 2,
            'sharedCatalog_getTaxClassId' => 1, 'storeRepository_getById' => 1,
            'sharedCatalog_getStoreId' => 2, 'sharedCatalog_getType' => 1, 'sharedCatalog_getAvailableTypes' => 1
        ];

        $taxClassId = 2634;
        $sharedCatalogType = 5;
        $returned = [
            'sharedCatalog_getTaxClassId' => $taxClassId, 'taxClass_getId' => $taxClassId,
            'sharedCatalog_getType' => $sharedCatalogType,
            'sharedCatalog_getAvailableTypes' => [$sharedCatalogType => $sharedCatalogType],
        ];

        $sharedCatalogId = 55;
        $this->sharedCatalog->expects($this->exactly(8))->method('getId')
            ->willReturnOnConsecutiveCalls(
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                null,
                $sharedCatalogId
            );
        $customerGroupId = 56;
        $this->sharedCatalog->expects($this->exactly(3))->method('getCustomerGroupId')->willReturn($customerGroupId);

        $this->prepareValidateMethod($returned, $calls);

        $this->assertNull($this->validator->validate($this->sharedCatalog));
    }

    /**
     * Test for validate() with InputException.
     *
     * @return void
     */
    public function testValidateWithInputException()
    {
        $this->expectException('Magento\Framework\Exception\InputException');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 1, 'sharedCatalogCollection_getFirstItem' => 1,
            'sharedCatalogCollectionFactory_create' => 1, 'sharedCatalog_getName' => 1,
            'sharedCatalog_getTaxClassId' => 1, 'storeRepository_getById' => 0,
            'sharedCatalog_getStoreId' => 0, 'sharedCatalog_getType' => 0, 'sharedCatalog_getAvailableTypes' => 0
        ];

        $taxClassId = 2634;
        $sharedCatalogTaxClassId = null;
        $sharedCatalogType = 5;
        $sharedCatalogAvailableTypes = [];
        $returned = [
            'sharedCatalog_getTaxClassId' => $sharedCatalogTaxClassId, 'taxClass_getId' => $taxClassId,
            'sharedCatalog_getType' => $sharedCatalogType,
            'sharedCatalog_getAvailableTypes' => $sharedCatalogAvailableTypes,
        ];

        $sharedCatalogId = 55;
        $this->sharedCatalog->expects($this->atLeastOnce())->method('getId')
            ->willReturnOnConsecutiveCalls($sharedCatalogId, $sharedCatalogId, $sharedCatalogId, null);

        $this->prepareValidateMethod($returned, $calls);

        $this->validator->validate($this->sharedCatalog);
    }

    /**
     * Test for validate() with NoSuchEntityException.
     *
     * @return void
     */
    public function testValidateWithNoSuchEntityException()
    {
        $this->expectException('Magento\Framework\Exception\NoSuchEntityException');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 1, 'sharedCatalogCollection_getFirstItem' => 1,
            'sharedCatalogCollectionFactory_create' => 1, 'sharedCatalog_getName' => 0,
            'sharedCatalog_getTaxClassId' => 0, 'storeRepository_getById' => 0,
            'sharedCatalog_getStoreId' => 0, 'sharedCatalog_getType' => 2, 'sharedCatalog_getAvailableTypes' => 1
        ];

        $taxClassId = 2634;
        $sharedCatalogType = 5;
        $sharedCatalogAvailableTypes = [7 => 7];
        $returned = [
            'sharedCatalog_getTaxClassId' => $taxClassId, 'taxClass_getId' => $taxClassId,
            'sharedCatalog_getType' => $sharedCatalogType,
            'sharedCatalog_getAvailableTypes' => $sharedCatalogAvailableTypes,
        ];

        $sharedCatalogId = 56;
        $this->sharedCatalog->expects($this->exactly(4))->method('getId')->willReturn($sharedCatalogId);

        $this->prepareValidateMethod($returned, $calls);

        $this->validator->validate($this->sharedCatalog);
    }

    /**
     * Test for validate() with NoSuchEntityException (SharedCatalogTaxClass validation).
     *
     * @return void
     */
    public function testValidateWithNoSuchEntityExceptionForTaxClass()
    {
        $this->expectException('Magento\Framework\Exception\NoSuchEntityException');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 1, 'sharedCatalogCollection_getFirstItem' => 1,
            'sharedCatalogCollectionFactory_create' => 1, 'sharedCatalog_getName' => 0,
            'sharedCatalog_getTaxClassId' => 2, 'storeRepository_getById' => 0,
            'sharedCatalog_getStoreId' => 0, 'sharedCatalog_getType' => 1, 'sharedCatalog_getAvailableTypes' => 1
        ];

        $taxClassId = 2634;
        $sharedCatalogType = 5;
        $sharedCatalogTaxClassId = 635;
        $sharedCatalogAvailableTypes = [$sharedCatalogType => $sharedCatalogType];
        $returned = [
            'sharedCatalog_getTaxClassId' => $sharedCatalogTaxClassId, 'taxClass_getId' => $taxClassId,
            'sharedCatalog_getType' => $sharedCatalogType,
            'sharedCatalog_getAvailableTypes' => $sharedCatalogAvailableTypes,
        ];

        $sharedCatalogId = 58;
        $this->sharedCatalog->expects($this->exactly(4))->method('getId')->willReturn($sharedCatalogId);

        $this->prepareValidateMethod($returned, $calls);

        $this->validator->validate($this->sharedCatalog);
    }

    /**
     * Test for validate() with InputException (SharedCatalogName validation).
     *
     * @return void
     */
    public function testValidateWithInputExceptionForSharedCatalogName()
    {
        $this->expectException('Magento\Framework\Exception\InputException');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 3, 'sharedCatalogCollection_getFirstItem' => 2,
            'sharedCatalogCollectionFactory_create' => 2, 'sharedCatalog_getName' => 3,
            'sharedCatalog_getTaxClassId' => 1, 'storeRepository_getById' => 1,
            'sharedCatalog_getStoreId' => 2, 'sharedCatalog_getType' => 1, 'sharedCatalog_getAvailableTypes' => 1
        ];

        $taxClassId = 2634;
        $sharedCatalogType = 5;
        $sharedCatalogAvailableTypes = [$sharedCatalogType => $sharedCatalogType];
        $returned = [
            'sharedCatalog_getTaxClassId' => $taxClassId, 'taxClass_getId' => $taxClassId,
            'sharedCatalog_getType' => $sharedCatalogType,
            'sharedCatalog_getAvailableTypes' => $sharedCatalogAvailableTypes,
        ];

        $sharedCatalogId = 59;
        $this->sharedCatalog->expects($this->exactly(7))->method('getId')->willReturn($sharedCatalogId);

        $this->prepareValidateMethod($returned, $calls);

        $this->validator->validate($this->sharedCatalog);
    }

    /**
     * Test for validate() with InputException (SharedCatalogName length validation).
     *
     * @return void
     */
    public function testValidateWithInputExceptionForSharedCatalogNameLength()
    {
        $this->expectException('Magento\Framework\Exception\InputException');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 1, 'sharedCatalogCollection_getFirstItem' => 1,
            'sharedCatalogCollectionFactory_create' => 1, 'sharedCatalog_getName' => 1,
            'sharedCatalog_getTaxClassId' => 1, 'storeRepository_getById' => 1,
            'sharedCatalog_getStoreId' => 2, 'sharedCatalog_getType' => 1, 'sharedCatalog_getAvailableTypes' => 1
        ];

        $taxClassId = 2634;
        $sharedCatalogType = 5;
        $sharedCatalogAvailableTypes = [$sharedCatalogType => $sharedCatalogType];
        $returned = [
            'sharedCatalog_getTaxClassId' => $taxClassId, 'taxClass_getId' => $taxClassId,
            'sharedCatalog_getType' => $sharedCatalogType,
            'sharedCatalog_getAvailableTypes' => $sharedCatalogAvailableTypes,
            'sharedCatalogName' => 'Very long name.........................................................'
        ];

        $sharedCatalogId = 1;
        $this->sharedCatalog->expects($this->any())->method('getId')->willReturn($sharedCatalogId);

        $this->prepareValidateMethod($returned, $calls);

        $this->validator->validate($this->sharedCatalog);
    }

    /**
     * Test for validate() with InputException (CustomerGroupChanges validation).
     *
     * @return void
     */
    public function testValidateWithInputExceptionForCustomerGroupChanges()
    {
        $this->expectException('Magento\Framework\Exception\InputException');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 4, 'sharedCatalogCollection_getFirstItem' => 3,
            'sharedCatalogCollectionFactory_create' => 3, 'sharedCatalog_getName' => 2,
            'sharedCatalog_getTaxClassId' => 1, 'storeRepository_getById' => 1,
            'sharedCatalog_getStoreId' => 2, 'sharedCatalog_getType' => 1, 'sharedCatalog_getAvailableTypes' => 1
        ];

        $taxClassId = 2634;
        $sharedCatalogType = 5;
        $sharedCatalogAvailableTypes = [$sharedCatalogType => $sharedCatalogType];
        $returned = [
            'sharedCatalog_getTaxClassId' => $taxClassId, 'taxClass_getId' => $taxClassId,
            'sharedCatalog_getType' => $sharedCatalogType,
            'sharedCatalog_getAvailableTypes' => $sharedCatalogAvailableTypes,
        ];

        $sharedCatalogId = 53;
        $this->sharedCatalog->expects($this->exactly(8))->method('getId')
            ->willReturnOnConsecutiveCalls(
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                $sharedCatalogId,
                null,
                $sharedCatalogId
            );
        $customerGroupId = 56;
        $this->sharedCatalog->expects($this->exactly(3))->method('getCustomerGroupId')
            ->willReturnOnConsecutiveCalls($customerGroupId, $customerGroupId, null);

        $this->prepareValidateMethod($returned, $calls);

        $this->validator->validate($this->sharedCatalog);
    }

    /**
     * Test for isCatalogPublicTypeDuplicated().
     *
     * @param bool $expects
     * @param array $returned
     * @param array $calls
     * @dataProvider isCatalogPublicTypeDuplicatedDataProvider
     * @return void
     */
    public function testIsCatalogPublicTypeDuplicated($expects, array $returned, array $calls)
    {
        $this->sharedCatalog->expects($this->once())->method('getType')
            ->willReturn($returned['sharedCatalog_getType']);

        $this->sharedCatalogManagement->expects($this->exactly($calls['sharedCatalogManagement_getPublicCatalog']))
            ->method('getPublicCatalog')
            ->willReturn($this->sharedCatalog);

        $sharedCatalogId = 5;
        $publicCatalogId = 34;
        $this->sharedCatalog->expects($this->exactly($calls['sharedCatalog_getId']))->method('getId')
            ->willReturnOnConsecutiveCalls($sharedCatalogId, $publicCatalogId);

        $result = $this->validator->isCatalogPublicTypeDuplicated($this->sharedCatalog);
        $this->assertEquals($expects, $result);
    }

    /**
     * Data provider for isCatalogPublicTypeDuplicated().
     *
     * @return array
     */
    public static function isCatalogPublicTypeDuplicatedDataProvider()
    {
        $sharedCatalogPublicType =  SharedCatalogInterface::TYPE_PUBLIC;
        return [
            [
                false, ['sharedCatalog_getType' => 9],
                ['sharedCatalogManagement_getPublicCatalog' => 0, 'sharedCatalog_getId' => 0]
            ],
            [
                true, ['sharedCatalog_getType' => $sharedCatalogPublicType],
                ['sharedCatalogManagement_getPublicCatalog' => 1, 'sharedCatalog_getId' => 2]
            ]
        ];
    }

    /**
     * Test for isCatalogPublicTypeDuplicated with NoSuchEntityException().
     *
     * @return void
     */
    public function testIsCatalogPublicTypeDuplicatedWithException()
    {
        $sharedCatalogPublicType =  SharedCatalogInterface::TYPE_PUBLIC;
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogPublicType);

        $exception = new NoSuchEntityException();
        $this->sharedCatalogManagement->expects($this->once())
            ->method('getPublicCatalog')
            ->willThrowException($exception);

        $expects = false;
        $result = $this->validator->isCatalogPublicTypeDuplicated($this->sharedCatalog);
        $this->assertEquals($expects, $result);
    }

    /**
     * Test for isDirectChangeToCustom().
     *
     * @param bool $expects
     * @param array $returned
     * @param array $calls
     * @dataProvider isDirectChangeToCustomDataProvider
     * @return void
     */
    public function testIsDirectChangeToCustom($expects, array $returned, array $calls)
    {
        $this->sharedCatalog->expects($this->once())->method('getType')
            ->willReturn($returned['sharedCatalog_getType']);

        $this->sharedCatalogManagement->expects($this->exactly($calls['sharedCatalogManagement_getPublicCatalog']))
            ->method('getPublicCatalog')
            ->willReturn($this->sharedCatalog);

        $sharedCatalogId = 5;
        $publicCatalogId = 34;
        $this->sharedCatalog->expects($this->exactly($calls['sharedCatalog_getId']))->method('getId')
            ->willReturnOnConsecutiveCalls($sharedCatalogId, $publicCatalogId);

        $result = $this->validator->isDirectChangeToCustom($this->sharedCatalog);
        $this->assertEquals($expects, $result);
    }

    /**
     * Data provider for isDirectChangeToCustom().
     *
     * @return array
     */
    public static function isDirectChangeToCustomDataProvider()
    {
        $sharedCatalogCustomType =  SharedCatalogInterface::TYPE_CUSTOM;
        return [
            [
                false, ['sharedCatalog_getType' => 9],
                ['sharedCatalogManagement_getPublicCatalog' => 0, 'sharedCatalog_getId' => 0]
            ],
            [
                true, ['sharedCatalog_getType' => $sharedCatalogCustomType],
                ['sharedCatalogManagement_getPublicCatalog' => 1, 'sharedCatalog_getId' => 2]
            ]
        ];
    }

    /**
     * Test for isDirectChangeToCustom() with NoSuchEntityException.
     *
     * @return void
     */
    public function testIsDirectChangeToCustomWithNoSuchEntityException()
    {
        $sharedCatalogCustomType =  SharedCatalogInterface::TYPE_CUSTOM;
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogCustomType);

        $exception = new NoSuchEntityException();
        $this->sharedCatalogManagement->expects($this->once())
            ->method('getPublicCatalog')
            ->willThrowException($exception);

        $expects = true;
        $result = $this->validator->isDirectChangeToCustom($this->sharedCatalog);
        $this->assertEquals($expects, $result);
    }

    /**
     * Test for isDirectChangeToCustom() with LocalizedException.
     *
     * @return void
     */
    public function testIsDirectChangeToCustomWithLocalizedException()
    {
        $this->expectException('Magento\Framework\Exception\LocalizedException');
        $sharedCatalogCustomType =  SharedCatalogInterface::TYPE_CUSTOM;
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogCustomType);

        $sharedCatalogId = 5;
        $this->sharedCatalog->expects($this->exactly(2))->method('getId')->willReturn($sharedCatalogId);

        $this->sharedCatalogManagement->expects($this->once())->method('getPublicCatalog')
            ->willReturn($this->sharedCatalog);

        $this->validator->isDirectChangeToCustom($this->sharedCatalog);
    }

    /**
     * Test checkSharedCatalogExist() with Exception.
     *
     * @return void
     */
    public function testCheckSharedCatalogExistWithException()
    {
        $this->expectException('Magento\Framework\Exception\NoSuchEntityException');
        $this->expectExceptionMessage('Requested Shared Catalog is not found');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 1, 'sharedCatalogCollection_getFirstItem' => 1,
            'sharedCatalogCollectionFactory_create' => 1
        ];

        $sharedCatalogId = 5;
        $this->sharedCatalog->expects($this->exactly(2))->method('getId')
            ->willReturnOnConsecutiveCalls($sharedCatalogId, null);

        $this->prepareSharedCatalogCollectionFactory($calls);

        $this->validator->checkSharedCatalogExist($this->sharedCatalog);
    }

    /**
     * Test for isSharedCatalogPublic().
     *
     * @return void
     */
    public function testIsSharedCatalogPublic()
    {
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 1, 'sharedCatalogCollection_getFirstItem' => 1,
            'sharedCatalogCollectionFactory_create' => 1
        ];

        $sharedCatalogId = 5;
        $this->sharedCatalog->expects($this->exactly(2))->method('getId')->willReturn($sharedCatalogId);

        $this->prepareSharedCatalogCollectionFactory($calls);

        $sharedCatalogCustomType =  SharedCatalogInterface::TYPE_CUSTOM;
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogCustomType);

        $expected = true;
        $result = $this->validator->isSharedCatalogPublic($this->sharedCatalog);
        $this->assertEquals($expected, $result);
    }

    /**
     * Test for isSharedCatalogPublic() with LocalizedException.
     *
     * @return void
     */
    public function testIsSharedCatalogPublicWithLocalizedException()
    {
        $this->expectException('Magento\Framework\Exception\LocalizedException');
        $calls = [
            'sharedCatalogCollection_addFieldToFilter' => 1, 'sharedCatalogCollection_getFirstItem' => 1,
            'sharedCatalogCollectionFactory_create' => 1
        ];

        $sharedCatalogId = 5;
        $this->sharedCatalog->expects($this->exactly(3))->method('getId')->willReturn($sharedCatalogId);

        $this->prepareSharedCatalogCollectionFactory($calls);

        $sharedCatalogPublicType =  SharedCatalogInterface::TYPE_PUBLIC;
        $this->sharedCatalog->expects($this->once())->method('getType')->willReturn($sharedCatalogPublicType);

        $this->validator->isSharedCatalogPublic($this->sharedCatalog);
    }

    /**
     * @return void
     * @throws NoSuchEntityException
     * @throws \Magento\Framework\Exception\InputException
     * @throws \PHPUnit\Framework\MockObject\Exception
     */
    public function testValidateThrowErrorOnCheckExistingSharedCatalogWithCustomerGroup(): void
    {
        $this->sharedCatalog->expects($this->once())->method('getId')->willReturn(null);
        $this->sharedCatalog->expects($this->exactly(2))->method('getCustomerGroupId')->willReturn(1);
        $collection = $this->createMock(\Magento\SharedCatalog\Model\ResourceModel\SharedCatalog\Collection::class);
        $collection->expects($this->once())
            ->method('addFieldToFilter')
            ->with($this->sharedCatalog::CUSTOMER_GROUP_ID, ['eq' => 1])
            ->willReturnSelf();
        $existingSharedCatalog = $this->createMock(SharedCatalog::class);
        $existingSharedCatalog->expects($this->once())->method('getId')->willReturn(2);
        $collection->expects($this->once())->method('getFirstItem')->willReturn($existingSharedCatalog);
        $this->sharedCatalogCollectionFactory->expects($this->once())->method('create')->willReturn($collection);
        $taxClassCollection = $this->getMockBuilder(Collection::class)
            ->onlyMethods(['getItems'])
            ->disableOriginalConstructor()
            ->getMock();
        $taxClassCollection->expects($this->once())->method('getItems')->willReturn([]);

        $this->taxClassCollectionFactory->expects($this->once())->method('create')
            ->willReturn($taxClassCollection);

        $this->expectException(InputException::class);
        $this->validator->validate($this->sharedCatalog);
    }
}
