<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\SharedCatalog\Model;

use Magento\Catalog\Model\ResourceModel\Product as ProductResourceModel;
use Magento\Customer\Api\Data\GroupInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\SharedCatalog\Api\CategoryManagementInterface;
use Magento\SharedCatalog\Api\Data\ProductItemInterface;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Api\ProductItemManagementInterface;
use Magento\SharedCatalog\Api\ProductItemRepositoryInterface;
use Magento\SharedCatalog\Api\ProductManagementInterface;
use Magento\SharedCatalog\Model\ResourceModel\CategoryProductLink;

/**
 * Shared catalog products actions.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class ProductManagement implements ProductManagementInterface
{
    /**
     * @var ProductItemManagementInterface
     */
    private $sharedCatalogProductItemManagement;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var ProductItemRepositoryInterface
     */
    private $sharedCatalogProductItemRepository;

    /**
     * @var SharedCatalogInvalidation
     */
    private $sharedCatalogInvalidation;

    /**
     * @var CategoryManagementInterface
     */
    private $sharedCatalogCategoryManagement;

    /**
     * @var CatalogPermissionManagement
     */
    private $catalogPermissionManagement;

    /**
     * Batch size to iterate collection
     *
     * @var int
     */
    private int $batchSize;

    /**
     * @var ProductResourceModel
     */
    private $productResourceModel;

    /**
     * @var CategoryProductLink
     */
    private $categoryProductLink;

    /**
     * @param ProductItemManagementInterface $productItemManagement
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param ProductItemRepositoryInterface $productItemRepository
     * @param SharedCatalogInvalidation $sharedCatalogInvalidation
     * @param CategoryManagementInterface $sharedCatalogCategoryManagement
     * @param CatalogPermissionManagement $catalogPermissionManagement
     * @param ProductResourceModel $productResourceModel
     * @param CategoryProductLink $categoryProductLink
     * @param int $batchSize defines how many items can be processed by one iteration
     */
    public function __construct(
        ProductItemManagementInterface $productItemManagement,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ProductItemRepositoryInterface $productItemRepository,
        SharedCatalogInvalidation $sharedCatalogInvalidation,
        CategoryManagementInterface $sharedCatalogCategoryManagement,
        CatalogPermissionManagement $catalogPermissionManagement,
        ProductResourceModel $productResourceModel,
        CategoryProductLink $categoryProductLink,
        int $batchSize = 200
    ) {
        $this->sharedCatalogProductItemManagement = $productItemManagement;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sharedCatalogProductItemRepository = $productItemRepository;
        $this->sharedCatalogInvalidation = $sharedCatalogInvalidation;
        $this->sharedCatalogCategoryManagement = $sharedCatalogCategoryManagement;
        $this->catalogPermissionManagement = $catalogPermissionManagement;
        $this->productResourceModel = $productResourceModel;
        $this->categoryProductLink = $categoryProductLink;
        $this->batchSize = $batchSize;
    }

    /**
     * @inheritdoc
     */
    public function getProducts($id)
    {
        $sharedCatalog = $this->sharedCatalogInvalidation->checkSharedCatalogExist($id);
        $this->searchCriteriaBuilder->addFilter(
            ProductItemInterface::CUSTOMER_GROUP_ID,
            $sharedCatalog->getCustomerGroupId()
        );
        $searchCriteria = $this->searchCriteriaBuilder->create();
        $searchCriteria->setPageSize($this->batchSize);

        $currentPage = 1;
        $productsSku = [];
        $totalCount = null;
        do {
            $searchCriteria->setCurrentPage($currentPage++);
            $searchResults = $this->sharedCatalogProductItemRepository->getList($searchCriteria);
            $productItems = $searchResults->getItems();
            if (count($productItems)) {
                $productsSku = array_merge($productsSku, $this->prepareProductSkus($productItems));
            }
            $totalCount = null === $totalCount
                ? $searchResults->getTotalCount() - $this->batchSize
                : $totalCount - $this->batchSize;
        } while ($totalCount > 0);

        return $productsSku;
    }

    /**
     * @inheritdoc
     */
    public function assignProducts($id, array $products)
    {
        $sharedCatalog = $this->sharedCatalogInvalidation->checkSharedCatalogExist($id);
        $customerGroupIds = $this->getAssociatedCustomerGroupIds($sharedCatalog);

        $skus = [];
        $ids = [];
        foreach ($products as $product) {
            if ($product->getSku()) {
                $skus[] = $product->getSku();
            } elseif ($product->getId()) {
                $ids[] = $product->getId();
            }
        }
        if (!empty($ids)) {
            $skus = array_merge($skus, array_column($this->productResourceModel->getProductsSku($ids), 'sku'));
        }
        $skus = array_unique($skus);
        $ids = [];
        if (!empty($skus)) {
            $ids = array_values($this->productResourceModel->getProductsIdsBySkus($skus));
        }

        $categoryIds = $this->sharedCatalogCategoryManagement->getCategories($sharedCatalog->getId());
        $productsCategoryIds = $this->categoryProductLink->getCategoryIds($ids);
        $assignCategoriesIds = array_diff($productsCategoryIds, $categoryIds);

        foreach ($customerGroupIds as $customerGroupId) {
            $this->sharedCatalogProductItemManagement->addItems($customerGroupId, $skus);
        }

        $this->sharedCatalogInvalidation->reindexCatalogProductPermissions($ids);
        $this->catalogPermissionManagement->setAllowPermissions($assignCategoriesIds, $customerGroupIds);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function unassignProducts($id, array $products)
    {
        $sharedCatalog = $this->sharedCatalogInvalidation->checkSharedCatalogExist($id);
        $skus = $this->sharedCatalogInvalidation->validateUnassignProducts($products);
        $customerGroupIds = $this->getAssociatedCustomerGroupIds($sharedCatalog);
        foreach ($customerGroupIds as $customerGroupId) {
            $this->deleteProductItems($customerGroupId, $skus, 'in');
        }

        return true;
    }

    /**
     * Reassign products to shared catalog.
     *
     * @param SharedCatalogInterface $sharedCatalog
     * @param array $skus
     * @return $this
     */
    public function reassignProducts(SharedCatalogInterface $sharedCatalog, array $skus)
    {
        $customerGroupIds = $this->getAssociatedCustomerGroupIds($sharedCatalog);
        foreach ($customerGroupIds as $customerGroupId) {
            if (!empty($skus)) {
                foreach (array_chunk($skus, $this->batchSize) as $skusBatch) {
                    $this->deleteProductItems($customerGroupId, $skusBatch);
                }
            } else {
                $this->deleteProductItems($customerGroupId, $skus);
            }
            $this->sharedCatalogProductItemManagement->addItems($customerGroupId, $skus);
        }

        return $this;
    }

    /**
     * Delete product items from shared catalog.
     *
     * @param int $customerGroupId
     * @param array $skus [optional]
     * @param string $conditionType [optional]
     * @return $this
     */
    private function deleteProductItems(int $customerGroupId, array $skus = [], string $conditionType = 'nin')
    {
        $this->searchCriteriaBuilder->setFilterGroups([]);
        $this->searchCriteriaBuilder->addFilter(ProductItemInterface::CUSTOMER_GROUP_ID, $customerGroupId);
        if (!empty($skus)) {
            $this->searchCriteriaBuilder->addFilter(ProductItemInterface::SKU, $skus, $conditionType);
        }
        $searchCriteria = $this->searchCriteriaBuilder->create();
        $productItems = $this->sharedCatalogProductItemRepository->getList($searchCriteria)->getItems();
        $this->sharedCatalogProductItemRepository->deleteItems($productItems);
        foreach ($productItems as $productItem) {
            $this->sharedCatalogInvalidation->cleanCacheByTag($productItem->getSku());
        }
        $this->sharedCatalogInvalidation->invalidateIndexRegistryItem();

        return $this;
    }

    /**
     * Prepare product skus array.
     *
     * @param ProductItemInterface[] $products
     * @return string[]
     */
    private function prepareProductSkus(array $products): array
    {
        $productsSkus = [];
        foreach ($products as $product) {
            $productsSkus[] = $product->getSku();
        }

        return $productsSkus;
    }

    /**
     * Get customer group ids that associated with shared catalog.
     *
     * @param SharedCatalogInterface $sharedCatalog
     * @return int[]
     */
    private function getAssociatedCustomerGroupIds(SharedCatalogInterface $sharedCatalog): array
    {
        $customerGroupIds = [(int) $sharedCatalog->getCustomerGroupId()];
        if ($sharedCatalog->getType() == SharedCatalogInterface::TYPE_PUBLIC) {
            $customerGroupIds[] = GroupInterface::NOT_LOGGED_IN_ID;
        }

        return $customerGroupIds;
    }
}
