<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Plugin\SharedCatalog;

use Magento\Authorization\Model\Role;
use Magento\Authorization\Model\UserContextInterface;
use Magento\Backend\Model\Auth\Session;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\App\State;
use Magento\SharedCatalog\Model\ResourceModel\SharedCatalog\Collection;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\ResourceModel\Website\CollectionFactory;
use Magento\User\Api\Data\UserInterfaceFactory;

/**
 * @SuppressWarnings(PHPMD.CookieAndSessionMisuse)
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CollectionFilter
{
    private const FILTERED_FLAG_NAME = 'admin_gws_filtered_b2b';

    private const INITIAL_FILTERED_FLAG = 'admin_gws_filtered';

    /**
     * @var array
     */
    private array $allWebsiteIds = [];

    /**
     * @var int|null
     */
    private ?int $adminId = null;

    /**
     * @var Role|null
     */
    private ?Role $adminRole = null;

    /**
     * @param CollectionFactory $websiteCollectionFactory
     * @param UserContextInterface $userContext
     * @param UserInterfaceFactory $userFactory
     * @param RequestInterface $request
     * @param Session $backendSession
     */
    public function __construct(
        private readonly CollectionFactory $websiteCollectionFactory,
        private readonly UserContextInterface $userContext,
        private readonly UserInterfaceFactory $userFactory,
        private readonly RequestInterface $request,
        private readonly Session $backendSession
    ) {
    }

    /**
     * Add filter to shared catalog collection.
     *
     * @param Collection $collection
     * @throws LocalizedException
     * @throws \Zend_Db_Select_Exception
     */
    private function filterCollection(Collection $collection): void
    {
        $role = $this->getAdminRole();
        if ($role && $role->getId() && !$role->getGwsIsAll() && !$collection->getFlag(self::FILTERED_FLAG_NAME)) {
            if (isset($collection->getSelect()->getPart(Select::FROM)['main_table'])) {
                if ($collection->getFlag(self::INITIAL_FILTERED_FLAG)) {
                    // the store_id from shared_catalog is actually store_group_id from store_group table,
                    // but the initial filtering assumes that
                    // store_id is actually from store table (as store view) not store_group table (as store)
                    $this->resetStoreGroupConditions($collection);
                }
                $this->filterSharedCatalogsByAllowedStoreGroupIdsAndAdminUser($collection, $role);
                $collection->setFlag(self::FILTERED_FLAG_NAME, true);
            }
        }
    }

    /**
     * Get admin user role for admin user context and webapi user context.
     *
     * @return Role|null
     * @throws LocalizedException
     */
    private function getAdminRole(): Role|null
    {
        if (!$this->adminRole) {
            if ($this->backendSession->getUser()) {
                $this->adminRole = $this->backendSession->getUser()->getRole();

            } elseif ($this->userContext->getUserId()
                && $this->userContext->getUserType() === UserContextInterface::USER_TYPE_ADMIN
            ) {
                $user = $this->userFactory->create();
                $user->load($this->userContext->getUserId());
                $this->adminRole = $user->getRole();
            }
        }
        return $this->adminRole;
    }

    /**
     * Get admin user id.
     *
     * @return int|null
     */
    private function getAdminId(): ?int
    {
        if (!$this->adminId) {
            $this->adminId = $this->userContext->getUserId();
        }
        return $this->adminId;
    }

    /**
     * Filter the shared catalog collection by allowed store ids.
     *
     * @param Collection $collection
     * @param Role $role
     * @return void
     * @throws \Zend_Db_Select_Exception
     */
    private function filterSharedCatalogsByAllowedStoreGroupIdsAndAdminUser(Collection $collection, Role $role): void
    {
        $allowedStoreGroupIds = $role->getGwsStoreGroups() ?? [];

        //restricted admin user has all websites assigned like full admin, so no need to filter
        $allowedWebsites = $role->getGwsWebsites() ?? [];
        if (count(array_diff($this->getAllWebsiteIds(), $allowedWebsites)) === 0) {
            $this->resetStoreGroupConditions($collection);
            return;
        }

        if ($this->request->getParam('shared_catalog_id')) {
            $collection->getSelect()->where(
                'main_table.entity_id = ?',
                (int)$this->request->getParam('shared_catalog_id')
            );
        } else {
            $adminUserId = $this->getAdminId();
            //filter by allowed store group ids when store_id is assigned to shared catalog,
            //or by created_by for shared catalog without assigned store group
            $collection->getSelect()
                ->where('main_table.store_id IN (?)', $allowedStoreGroupIds, \Zend_Db::INT_TYPE)
                ->orWhere(
                    '(main_table.store_id IS NULL AND main_table.created_by = ?)',
                    $adminUserId,
                    \Zend_Db::INT_TYPE
                );
        }
    }

    /**
     * Reset store where conditions
     *
     * @param Collection $collection
     * @return void
     * @throws \Zend_Db_Select_Exception
     */
    private function resetStoreGroupConditions(Collection $collection): void
    {
        $where = $collection->getSelect()->getPart(Select::WHERE);
        foreach ($where as $key => $value) {
            if (str_contains($value, 'main_table.store_id')) {
                unset($where[$key]);
            }
        }
        $collection->getSelect()->setPart(Select::WHERE, $where);
    }

    /**
     * Get all websites ids excluding admin store.
     *
     * @return array
     */
    private function getAllWebsiteIds(): array
    {
        if (!count($this->allWebsiteIds)) {
            $this->allWebsiteIds = array_filter($this->websiteCollectionFactory->create()->getAllIds());
        }

        return  $this->allWebsiteIds;
    }

    /**
     * Adds only allowed stores to shared catalog filter count.
     *
     * @param Collection $collection
     * @throws LocalizedException
     * @throws \Zend_Db_Select_Exception
     */
    public function beforeGetSelectCountSql(Collection $collection): void
    {
        $this->filterCollection($collection);
    }

    /**
     * Adds only restricted admin allowed stores to shared catalog filter.
     *
     * @param Collection $collection
     * @param bool $printQuery
     * @param bool $logQuery
     * @return void
     * @throws LocalizedException
     * @throws \Zend_Db_Select_Exception
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function beforeLoadWithFilter(
        Collection $collection,
        bool       $printQuery = false,
        bool       $logQuery = false
    ): void {
        $this->filterCollection($collection);
    }
}
