<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Fixture;

use Magento\Framework\App\ObjectManager;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\InvalidArgumentException;
use Magento\Framework\MessageQueue\ConsumerFactory;
use Magento\SharedCatalog\Model\CatalogPermissionManagement;
use Magento\SharedCatalog\Model\ResourceModel\Permission\CategoryPermissions\ScheduleBulk;
use Magento\TestFramework\Fixture\DataFixtureInterface;

class SetPermission implements DataFixtureInterface
{
    private const CATALOG = 'catalog';
    private const CATEGORY_ID = 'category_id';

    /**
     * @param ObjectManager $objectManager
     * @param CatalogPermissionManagement $catalogPermissionManagement
     */
    public function __construct(
        private ObjectManager $objectManager,
        private CatalogPermissionManagement $catalogPermissionManagement,
        private ScheduleBulk $scheduleBulk
    ) {
    }

    /**
     * @inheritdoc
     */
    public function apply(array $data = []): ?DataObject
    {
        if (empty($data[self::CATEGORY_ID])) {
            throw new InvalidArgumentException(__('"%field" is required', ['field' => self::CATEGORY_ID]));
        }

        if (empty($data[self::CATALOG])) {
            throw new InvalidArgumentException(__('"%field" is required', ['field' => self::CATALOG]));
        }

        $customerGroupId = $data[self::CATALOG]->getCustomerGroupId();

        if ($data['type'] === 'deny') {
            $this->catalogPermissionManagement->setDenyPermissions(
                [$data[self::CATEGORY_ID]],
                [$customerGroupId]
            );
        } else {
            $this->catalogPermissionManagement->setAllowPermissions(
                [$data[self::CATEGORY_ID]],
                [$customerGroupId]
            );
        }
        $this->scheduleBulk->execute([$data[self::CATEGORY_ID]], [$customerGroupId]);

        $this->queueConsumerStart('sharedCatalogUpdateCategoryPermissions');
        $this->queueConsumerStart('sharedCatalogUpdatePrice');

        return null;
    }

    /**
     * Process queue job for catalog permissions
     *
     * @param string $consumerName
     * @param int $maxNumberOfMessages
     */
    private function queueConsumerStart(string $consumerName, int $maxNumberOfMessages = 1000): void
    {
        /** @var \Magento\Framework\MessageQueue\ConsumerFactory $consumerFactory */
        $consumerFactory = $this->objectManager->get(ConsumerFactory::class);
        $categoryPermissionsUpdater = $consumerFactory->get($consumerName);
        $categoryPermissionsUpdater->process($maxNumberOfMessages);
    }
}
