<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Plugin\Ui\DataProvider;

use Magento\Authorization\Model\Role;
use Magento\Backend\Model\Auth\Session;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\SharedCatalog\Ui\DataProvider\Website as Websites;
use Magento\Store\Model\ResourceModel\Website\CollectionFactory;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Plugin for store switch permission based on role
 *
 * @SuppressWarnings(PHPMD.CookieAndSessionMisuse)
 */
class WebsiteRolePermissions
{
    /**
     * @var ArrayManager
     */
    private $arrayManager;

    /**
     * @var Session
     */
    private $backendAuthSession;

    /**
     * @var array
     */
    private $allWebsiteIds = [];

    /**
     * @param Session $backendAuthSession
     * @param ArrayManager $arrayManager
     * @param CollectionFactory $websiteCollectionFactory
     */
    public function __construct(
        Session $backendAuthSession,
        ArrayManager $arrayManager,
        private readonly CollectionFactory $websiteCollectionFactory
    ) {
        $this->backendAuthSession = $backendAuthSession;
        $this->arrayManager = $arrayManager;
    }

    /**
     * Remove 'All Stores' for website restricted users
     *
     * @param Websites $subject
     * @param array $result
     * @return array
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterGetWebsites(
        Websites $subject,
        array    $result
    ):array {
        $role = $this->backendAuthSession->getUser()->getRole();
        $allowedWebsites = $role->getGwsWebsites() ?? [];
        if (!$role->getGwsIsAll() &&
            count(array_diff($this->getAllWebsiteIds(), $allowedWebsites)) !== 0 &&
            $this->arrayManager->exists(0, $result)) {
            array_shift($result);
        }
        return $result;
    }

    /**
     * Get all website ids excluding admin store.
     *
     * @return array
     */
    private function getAllWebsiteIds(): array
    {
        if (!count($this->allWebsiteIds)) {
            $this->allWebsiteIds = array_filter($this->websiteCollectionFactory->create()->getAllIds());
        }

        return  $this->allWebsiteIds;
    }
}
