<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Plugin\Catalog\Model\ResourceModel\Product\Indexer;

use Magento\Catalog\Model\ResourceModel\Product\BaseSelectProcessorInterface;
use Magento\Catalog\Model\ResourceModel\Product\Indexer\LinkedProductSelectBuilderByIndexPrice;
use Magento\Company\Model\CompanyContextFactory;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\SharedCatalog\Model\CustomerGroupManagement;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use \Magento\SharedCatalog\Model\Config;

class LinkedProductSelectBuilderByIndexPricePlugin
{
    /**
     * @param CompanyContextFactory $companyContextFactory
     * @param StoreManagerInterface $storeManager
     * @param Config $config
     * @param CustomerGroupManagement $customerGroupManagement
     * @param ResourceConnection $resourceConnection
     */
    public function __construct(
        private readonly CompanyContextFactory $companyContextFactory,
        private readonly StoreManagerInterface $storeManager,
        private readonly Config $config,
        private readonly CustomerGroupManagement $customerGroupManagement,
        private readonly ResourceConnection $resourceConnection
    ) {
    }

    /**
     * Filter child products by shared catalog
     *
     * @param LinkedProductSelectBuilderByIndexPrice $subject
     * @param array $result
     * @param int $productId
     * @param int $storeId
     * @return array
     * @throws LocalizedException
     * @throws NoSuchEntityException
     * @SuppressWarnings(PHPMD.UnusedFormalParameters)
     */
    public function afterBuild(
        LinkedProductSelectBuilderByIndexPrice $subject,
        array $result,
        int $productId,
        int $storeId
    ): array {
        $companyContext = $this->companyContextFactory->create();
        $customerGroupId = $companyContext->getCustomerGroupId();
        $websiteId = $this->storeManager->getStore($storeId)->getWebsiteId();

        if (($this->config->isActive(ScopeInterface::SCOPE_WEBSITE, $websiteId)
            && !$this->customerGroupManagement->isPrimaryCatalogAvailable($customerGroupId)) === false
        ) {
            return $result;
        }

        $sharedProductTable = $this->resourceConnection->getTableName('shared_catalog_product_item');

        /** @var Select $select */
        foreach ($result as $select) {
            $select->joinInner(
                ['shared_product' => $sharedProductTable],
                'shared_product.sku = ' . BaseSelectProcessorInterface::PRODUCT_TABLE_ALIAS .
                '.sku AND shared_product.customer_group_id = ' . $customerGroupId,
                []
            );
        }

        return $result;
    }
}
