<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Plugin\Catalog\Model\Product;

use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\Copier;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager as ObjectManagerHelper;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\SharedCatalog\Api\ProductItemManagementInterface;
use Magento\SharedCatalog\Model\ProductSharedCatalogsLoader;
use Magento\SharedCatalog\Plugin\Catalog\Model\Product\AssignSharedCatalogOnDuplicateProductPlugin;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * Unit tests for AssignSharedCatalogOnDuplicateProductPlugin plugin.
 */
class AssignSharedCatalogOnDuplicateProductPluginTest extends TestCase
{
    /**
     * @var ObjectManagerHelper
     */
    private $objectManagerHelper;

    /**
     * @var AssignSharedCatalogOnDuplicateProductPlugin|MockObject
     */
    private $assignSharedCatalogOnDuplicateProductPlugin;

    /**
     * @var ProductSharedCatalogsLoader|MockObject
     */
    private $productSharedCatalogsLoaderMock;

    /**
     * @var ProductItemManagementInterface|MockObject
     */
    private $productItemManagementMock;

    /**
     * @var LoggerInterface|MockObject
     */
    private $loggerMock;

    /**
     * Set up.
     *
     * @return void
     */
    protected function setUp(): void
    {
        $this->productSharedCatalogsLoaderMock = $this->getMockBuilder(
            ProductSharedCatalogsLoader::class
        )
            ->disableOriginalConstructor()
            ->getMock();
        $this->productItemManagementMock = $this->getMockBuilder(
            ProductItemManagementInterface::class
        )
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->loggerMock = $this->getMockBuilder(LoggerInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $this->objectManagerHelper = new ObjectManagerHelper($this);
        $this->assignSharedCatalogOnDuplicateProductPlugin = $this->objectManagerHelper->getObject(
            AssignSharedCatalogOnDuplicateProductPlugin::class,
            [
                'productSharedCatalogsLoader' => $this->productSharedCatalogsLoaderMock,
                'productItemManagement' => $this->productItemManagementMock,
                'logger' => $this->loggerMock
            ]
        );
    }

    /**
     * Test for aroundCopy() method.
     *
     * @return void
     */
    public function testAfterCopy()
    {
        $sku = 'sku';
        $customerGroupId = 1;

        $productMock = $this->getMockBuilder(Product::class)
            ->disableOriginalConstructor()
            ->getMock();
        $productMock->expects($this->atLeastOnce())->method('getSku')->willReturn($sku);
        $sharedCatalogMock = $this->getMockBuilder(SharedCatalogInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->productSharedCatalogsLoaderMock->expects($this->once())->method('getAssignedSharedCatalogs')
            ->willReturn([$sharedCatalogMock]);
        $sharedCatalogMock->expects($this->atLeastOnce())->method('getCustomerGroupId')->willReturn($customerGroupId);

        $exception = new LocalizedException(__('test'));
        $this->productItemManagementMock->expects($this->once())->method('addItems')->with($customerGroupId, [$sku])
            ->willThrowException($exception);
        $this->loggerMock->expects($this->once())->method('critical')->with($exception);

        $subject = $this->getMockBuilder(Copier::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->assertEquals(
            $productMock,
            $this->assignSharedCatalogOnDuplicateProductPlugin->afterCopy($subject, $productMock, $productMock)
        );
    }
}
