<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Test\Unit\Plugin\Catalog\Model;

/**
 * Unit test for Magento\SharedCatalog\Plugin\Catalog\Model\DenyPermissionsForNewCategory class.
 */
class DenyPermissionsForNewCategoryTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var \Magento\SharedCatalog\Model\Config\CategoryPermission|\PHPUnit_Framework_MockObject_MockObject
     */
    private $configCategoryPermission;

    /**
     * @var \Magento\SharedCatalog\Model\CatalogPermissionManagement|\PHPUnit_Framework_MockObject_MockObject
     */
    private $catalogPermissionManagement;

    /**
     * @var \Magento\SharedCatalog\Plugin\Catalog\Model\DenyPermissionsForNewCategory
     */
    private $plugin;

    /**
     * @var \Magento\Catalog\Model\Category|\PHPUnit_Framework_MockObject_MockObject
     */
    private $subject;

    /**
     * @var \Magento\Catalog\Model\Category|\PHPUnit_Framework_MockObject_MockObject
     */
    private $category;

    /**
     * @inheritdoc
     */
    protected function setUp()
    {
        $this->configCategoryPermission = $this->getMockBuilder(
            \Magento\SharedCatalog\Model\Config\CategoryPermission::class
        )
            ->disableOriginalConstructor()
            ->getMock();
        $this->catalogPermissionManagement = $this->getMockBuilder(
            \Magento\SharedCatalog\Model\CatalogPermissionManagement::class
        )
            ->disableOriginalConstructor()
            ->getMock();
        $this->subject = $this->createMock(\Magento\Catalog\Model\Category::class);
        $this->category = $this->createPartialMock(
            \Magento\Catalog\Model\Category::class,
            ['getPermissions', 'getId', 'isObjectNew']
        );

        $objectManager = new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);
        $this->plugin = $objectManager->getObject(
            \Magento\SharedCatalog\Plugin\Catalog\Model\DenyPermissionsForNewCategory::class,
            [
                'configCategoryPermission' => $this->configCategoryPermission,
                'catalogPermissionManagement' => $this->catalogPermissionManagement,
            ]
        );
    }

    /**
     * Test afterSave method.
     *
     * @return void
     */
    public function testAfterSave()
    {
        $categoryId = 1;
        $this->category->expects($this->once())->method('isObjectNew')->willReturn(true);
        $this->configCategoryPermission->expects($this->once())->method('isConfigEnable')->willReturn(true);
        $this->category->expects($this->once())->method('getPermissions')->willReturn([]);
        $this->category->expects($this->once())->method('getId')->willReturn($categoryId);
        $this->catalogPermissionManagement->expects($this->once())
            ->method('setDenyPermissionsForCategory')
            ->with($categoryId);

        $this->assertEquals($this->category, $this->plugin->afterSave($this->subject, $this->category));
    }

    /**
     * Test afterSave method when config is disabled.
     *
     * @return void
     */
    public function testAfterSaveWhenDisabledInConfig()
    {
        $this->category->expects($this->once())->method('isObjectNew')->willReturn(true);
        $this->configCategoryPermission->expects($this->once())->method('isConfigEnable')->willReturn(false);
        $this->category->expects($this->once())->method('getPermissions')->willReturn([]);
        $this->catalogPermissionManagement->expects($this->never())->method('setDenyPermissionsForCategory');

        $this->assertEquals($this->category, $this->plugin->afterSave($this->subject, $this->category));
    }

    /**
     * Test afterSave method when category is not new.
     *
     * @return void
     */
    public function testAfterSaveWhenCategoryIsNotNew(): void
    {
        $this->category->expects($this->once())->method('isObjectNew')->willReturn(false);
        $this->configCategoryPermission->expects($this->never())->method('isConfigEnable');
        $this->category->expects($this->never())->method('getPermissions');
        $this->category->expects($this->never())->method('getId');
        $this->catalogPermissionManagement->expects($this->never())->method('setDenyPermissionsForCategory');

        $this->assertEquals($this->category, $this->plugin->afterSave($this->subject, $this->category));
    }
}
