<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\SharedCatalog\Test\Unit\Observer;

use Magento\Framework\App\Config\ConfigResource\ConfigInterface;

/**
 * Class UpdateConfigTest
 */
class UpdateConfigTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var \Magento\Store\Model\StoreManagerInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    protected $storeManager;

    /**
     * @var \Magento\Company\Api\StatusServiceInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    protected $companyStatusService;

    /**
     * @var \Magento\SharedCatalog\Model\Config|\PHPUnit_Framework_MockObject_MockObject
     */
    protected $sharedCatalogModuleConfig;

    /**
     * @var \Magento\SharedCatalog\Model\CategoryPermissions|\PHPUnit_Framework_MockObject_MockObject
     */
    private $categoryPermissions;

    /**
     * @var \Magento\Framework\App\Config\ConfigResource\ConfigInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $configResourceMock;

    /**
     * @var \Magento\Framework\Event\Observer|\PHPUnit_Framework_MockObject_MockObject
     */
    protected $observer;

    /**
     * @var \Magento\Framework\Event|\PHPUnit_Framework_MockObject_MockObject
     */
    protected $event;

    /**
     * @var \Magento\SharedCatalog\Observer\UpdateConfig|\PHPUnit_Framework_MockObject_MockObject
     */
    protected $updateConfig;

    /**
     * @var \Magento\SharedCatalog\Model\CatalogPermissionManagement|\PHPUnit_Framework_MockObject_MockObject
     */
    private $catalogPermissionsManagement;

    /**
     * Set up
     */
    protected function setUp()
    {
        $this->storeManager = $this->createMock(\Magento\Store\Model\StoreManagerInterface::class);
        $this->companyStatusService = $this->createMock(\Magento\Company\Api\StatusServiceInterface::class);
        $this->sharedCatalogModuleConfig = $this->createMock(\Magento\SharedCatalog\Model\Config::class);
        $this->categoryPermissions = $this->getMockBuilder(\Magento\SharedCatalog\Model\CategoryPermissions::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->configResourceMock = $this->getMockBuilder(ConfigInterface::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->observer = $this->getMockBuilder(\Magento\Framework\Event\Observer::class)
            ->disableOriginalConstructor()->getMock();
        $this->event = $this->getMockBuilder(\Magento\Framework\Event::class)
            ->setMethods(['getWebsite', 'getData'])
            ->disableOriginalConstructor()->getMock();
        $this->observer->expects($this->any())->method('getEvent')
            ->willReturn($this->event);
        $this->catalogPermissionsManagement = $this
            ->getMockBuilder(\Magento\SharedCatalog\Model\CatalogPermissionManagement::class)
            ->disableOriginalConstructor()
            ->getMock();

        $objectManager = new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);
        $this->updateConfig = $objectManager->getObject(
            \Magento\SharedCatalog\Observer\UpdateConfig::class,
            [
                'storeManager' => $this->storeManager,
                'companyStatusService' => $this->companyStatusService,
                'sharedCatalogModuleConfig' => $this->sharedCatalogModuleConfig,
                'categoryPermissions' => $this->categoryPermissions,
                'catalogPermissionsManagement' => $this->catalogPermissionsManagement,
                'configResource' => $this->configResourceMock,
            ]
        );
    }

    /**
     * @param int $eventWebsiteId
     * @param bool $isCompanyActive
     * @param bool $isSharedCatalogActive
     * @param int $enablingCalls
     * @param int $disablingCalls
     * @return void
     * @dataProvider dataProviderExecute
     */
    public function testExecute(
        int $eventWebsiteId,
        bool $isCompanyActive,
        bool $isSharedCatalogActive,
        int $enablingCalls,
        int $disablingCalls
    ) {
        $this->event->expects($this->atLeastOnce())
            ->method('getWebsite')
            ->willReturn($eventWebsiteId);
        $this->event->expects($this->once())
            ->method('getData')
            ->with('changed_paths')
            ->willReturn([
                'btob/website_configuration/company_active',
                \Magento\SharedCatalog\Model\Config::CONFIG_SHARED_CATALOG,
            ]);

        $website = $this->getMockBuilder(\Magento\Store\Api\Data\WebsiteInterface::class)
            ->getMock();
        $this->storeManager->expects($this->any())
            ->method('getWebsite')
            ->willReturn($website);

        $this->companyStatusService->expects($this->once())
            ->method('isActive')
            ->willReturn($isCompanyActive);

        $this->sharedCatalogModuleConfig->expects($this->any())
            ->method('isActive')
            ->willReturn($isSharedCatalogActive);

        $isRequireModuleDisable = !$isCompanyActive && $isSharedCatalogActive;
        $this->configResourceMock->expects(
            $this->exactly($isRequireModuleDisable ? 1 : 0)
        )->method('saveConfig');

        $this->catalogPermissionsManagement->expects($this->exactly($enablingCalls))
            ->method('setPermissionsForAllCategories');
        $this->categoryPermissions->expects($this->exactly($enablingCalls))
            ->method('enable');
        $this->catalogPermissionsManagement->expects($this->exactly($disablingCalls))
            ->method('removePermissionsForAllCategories');
        $this->categoryPermissions->expects($this->exactly($disablingCalls))
            ->method('disable');

        $this->updateConfig->execute($this->observer);
    }

    /**
     * @return array
     */
    public function dataProviderExecute(): array
    {
        return [
            [1, true, true, 1, 0],
            [0, false, true, 0, 1],
            [1, false, false, 0, 1],
            [0, true, false, 0, 1],
        ];
    }
}
