<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\SharedCatalog\Test\Unit\Model;

use PHPUnit_Framework_MockObject_MockObject as MockObject;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\SharedCatalog\Model\SharedCatalogInvalidation;
use Magento\SharedCatalog\Model\CatalogPermissionManagement;
use Magento\Store\Model\StoreManagerInterface;
use Magento\SharedCatalog\Model\SharedCatalogAssignment;
use Magento\CatalogPermissions\Helper\Data as HelperData;
use Magento\SharedCatalog\Model\CategoryManagement;
use Magento\Framework\TestFramework\Unit\Helper\ObjectManager;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\Store\Api\Data\GroupInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Model\Category;
use Magento\Catalog\Model\ResourceModel\Category\Collection;
use Magento\Customer\Api\Data\GroupInterface as CustomerGroupInterface;
use Magento\Store\Model\Store;

/**
 * CategoryManagement unit test.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CategoryManagementTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var CategoryRepositoryInterface|MockObject
     */
    private $categoryRepository;

    /**
     * @var SharedCatalogInvalidation|MockObject
     */
    private $sharedCatalogInvalidation;

    /**
     * @var CatalogPermissionManagement|MockObject
     */
    private $catalogPermissionManagement;

    /**
     * @var StoreManagerInterface|MockObject
     */
    private $storeManager;

    /**
     * @var SharedCatalogAssignment|MockObject
     */
    private $sharedCatalogAssignment;

    /**
     * @var Data|MockObject
     */
    private $permissionsHelper;

    /**
     * @var CategoryManagement
     */
    private $categoryManagement;

    /**
     * Set up.
     *
     * @return void
     */
    protected function setUp()
    {
        $this->categoryRepository = $this->getMockBuilder(CategoryRepositoryInterface::class)
            ->disableOriginalConstructor()->getMockForAbstractClass();
        $this->sharedCatalogInvalidation = $this
            ->getMockBuilder(SharedCatalogInvalidation::class)
            ->disableOriginalConstructor()->getMock();
        $this->catalogPermissionManagement = $this
            ->getMockBuilder(CatalogPermissionManagement::class)
            ->disableOriginalConstructor()->getMock();
        $this->storeManager = $this->getMockBuilder(StoreManagerInterface::class)
            ->disableOriginalConstructor()->getMockForAbstractClass();
        $this->sharedCatalogAssignment = $this
            ->getMockBuilder(SharedCatalogAssignment::class)
            ->disableOriginalConstructor()->getMock();
        $this->permissionsHelper = $this
            ->getMockBuilder(HelperData::class)
            ->disableOriginalConstructor()->getMock();

        $objectManager = new ObjectManager($this);
        $this->categoryManagement = $objectManager->getObject(
            CategoryManagement::class,
            [
                'categoryRepository' => $this->categoryRepository,
                'sharedCatalogInvalidation' => $this->sharedCatalogInvalidation,
                'catalogPermissionManagement' => $this->catalogPermissionManagement,
                'storeManager' => $this->storeManager,
                'sharedCatalogAssignment' => $this->sharedCatalogAssignment,
                'permissionsHelper' => $this->permissionsHelper,
            ]
        );
    }

    /**
     * Test for getCategories method.
     *
     * @param int|null $storeId
     * @param int $rootCategoryId
     * @param array $expectedResult
     * @return void
     * @dataProvider getCategoriesDataProvider
     */
    public function testGetCategories(
        $storeId,
        $rootCategoryId,
        array $expectedResult
    ) {
        $sharedCatalogId = 1;
        $websiteId = 7;

        $sharedCatalog = $this->createMock(SharedCatalogInterface::class);
        $this->sharedCatalogInvalidation->method('checkSharedCatalogExist')
            ->with($sharedCatalogId)
            ->willReturn($sharedCatalog);
        $sharedCatalog->method('getStoreId')
            ->willReturn($storeId);

        $store = $this->createMock(GroupInterface::class);
        $this->storeManager->method('getGroup')
            ->with((int)$storeId)
            ->willReturn($store);
        $store->method('getRootCategoryId')
            ->willReturn($rootCategoryId);

        $category = $this->createMock(Category::class);
        $category->method('getId')
            ->willReturn($rootCategoryId);
        $category->method('getPath')
            ->willReturn($rootCategoryId);
        $this->categoryRepository->method('get')
            ->with($rootCategoryId)
            ->willReturn($category);
        $categoryCollection = $this->createMock(Collection::class);
        $category->method('getCollection')
            ->willReturn($categoryCollection);
        $categoryCollection->method('addPathsFilter')
            ->with($rootCategoryId . '/')
            ->willReturnSelf();
        $categoryCollection->method('getAllIds')
            ->willReturn([4, 5, 6]);

        $store->method('getWebsiteId')
            ->willReturn($websiteId);
        $this->catalogPermissionManagement->method('getAllowedCategoriesIds')
            ->with($sharedCatalogId, $websiteId)
            ->willReturn([6]);

        $this->assertEquals($expectedResult, $this->categoryManagement->getCategories($sharedCatalogId));
    }

    /**
     * Test for assignCategories method.
     *
     * @return void
     */
    public function testAssignCategories()
    {
        $sharedCatalogId = 1;
        $categoryId = 2;
        $customerGroupId = 5;

        $sharedCatalog = $this->createMock(SharedCatalogInterface::class);
        $this->sharedCatalogInvalidation->method('checkSharedCatalogExist')
            ->with($sharedCatalogId)
            ->willReturn($sharedCatalog);

        $category = $this->createMock(CategoryInterface::class);
        $category->method('getId')
            ->willReturn($categoryId);

        $store = $this->createMock(GroupInterface::class);
        $this->storeManager->method('getGroup')
            ->with(Store::DEFAULT_STORE_ID)
            ->willReturn($store);

        $rootCategory = $this->createMock(Category::class);
        $rootCategory->method('getId')
            ->willReturn(1);
        $rootCategory->method('getPath')
            ->willReturn(1);
        $this->categoryRepository->method('get')
            ->with(1)
            ->willReturn($rootCategory);
        $categoryCollection = $this->createMock(Collection::class);
        $rootCategory->method('getCollection')
            ->willReturn($categoryCollection);
        $categoryCollection->method('addPathsFilter')
            ->with('1/')
            ->willReturnSelf();
        $categoryCollection->method('getAllIds')
            ->willReturn([2, 3, 4]);

        $sharedCatalog->method('getCustomerGroupId')
            ->willReturn($customerGroupId);
        $this->catalogPermissionManagement->method('setAllowPermissions')
            ->with([$categoryId], [$customerGroupId]);

        $this->assertTrue($this->categoryManagement->assignCategories($sharedCatalogId, [$category]));
    }

    /**
     * Test for assignCategories method with exception.
     *
     * @return void
     * @expectedException \Magento\Framework\Exception\NoSuchEntityException
     * @expectedExceptionMessage Requested categories don't exist: 2
     */
    public function testAssignCategoriesWithException()
    {
        $sharedCatalogId = 1;
        $categoryId = 2;

        $sharedCatalog = $this->createMock(SharedCatalogInterface::class);
        $this->sharedCatalogInvalidation->method('checkSharedCatalogExist')
            ->with($sharedCatalogId)
            ->willReturn($sharedCatalog);

        $category = $this->createMock(CategoryInterface::class);
        $category->method('getId')
            ->willReturn($categoryId);

        $store = $this->createMock(GroupInterface::class);
        $this->storeManager->method('getGroup')
            ->with(Store::DEFAULT_STORE_ID)
            ->willReturn($store);

        $rootCategory = $this->createMock(Category::class);
        $rootCategory->method('getId')
            ->willReturn(1);
        $rootCategory->method('getPath')
            ->willReturn(1);
        $this->categoryRepository->method('get')
            ->with(1)
            ->willReturn($rootCategory);
        $categoryCollection = $this->createMock(Collection::class);
        $rootCategory->method('getCollection')
            ->willReturn($categoryCollection);
        $categoryCollection->method('addPathsFilter')
            ->with('1/')
            ->willReturnSelf();
        $categoryCollection->method('getAllIds')
            ->willReturn([3, 4]);

        $this->categoryManagement->assignCategories($sharedCatalogId, [$category]);
    }

    /**
     * Test for unassignCategories method.
     *
     * @return void
     */
    public function testUnassignCategories()
    {
        $sharedCatalogId = 1;
        $categoryId = 2;
        $customerGroupId = 5;
        $storeId = 6;
        $rootCategoryId = 7;
        $websiteId = 8;

        $sharedCatalog = $this->createMock(SharedCatalogInterface::class);
        $this->sharedCatalogInvalidation->method('checkSharedCatalogExist')
            ->with($sharedCatalogId)
            ->willReturn($sharedCatalog);

        $category = $this->createMock(CategoryInterface::class);
        $category->method('getId')
            ->willReturn($categoryId);

        $store = $this->createMock(GroupInterface::class);
        $this->storeManager->method('getGroup')
            ->with($storeId)
            ->willReturn($store);

        $rootCategory = $this->createMock(Category::class);
        $rootCategory->method('getId')
            ->willReturnOnConsecutiveCalls(1, $rootCategoryId);
        $rootCategory->method('getPath')
            ->willReturnOnConsecutiveCalls(1, $rootCategoryId);
        $this->categoryRepository->method('get')
            ->withConsecutive([1], [$rootCategoryId])
            ->willReturn($rootCategory);
        $categoryCollection = $this->createMock(Collection::class);
        $rootCategory->method('getCollection')
            ->willReturn($categoryCollection);
        $categoryCollection->method('addPathsFilter')
            ->withConsecutive(['1/'], [$rootCategoryId . '/'])
            ->willReturnSelf();
        $categoryCollection->method('getAllIds')
            ->willReturn([2, 3, 4]);

        $sharedCatalog->method('getCustomerGroupId')
            ->willReturn($customerGroupId);
        $sharedCatalog->method('getType')
            ->willReturn(SharedCatalogInterface::TYPE_PUBLIC);
        $this->catalogPermissionManagement->method('setDenyPermissions')
            ->with([$categoryId], [$customerGroupId, CustomerGroupInterface::NOT_LOGGED_IN_ID]);
        $this->sharedCatalogAssignment->method('unassignProductsForCategories')
            ->with($sharedCatalogId, [$categoryId]);
        $sharedCatalog->method('getStoreId')
            ->willReturn($storeId);
        $store->method('getRootCategoryId')
            ->willReturn($rootCategoryId);
        $store->method('getWebsiteId')
            ->willReturn($websiteId);
        $this->catalogPermissionManagement->method('getAllowedCategoriesIds')
            ->with($sharedCatalogId, $websiteId)
            ->willReturn([3]);

        $this->assertTrue($this->categoryManagement->unassignCategories($sharedCatalogId, [$category]));
    }

    /**
     * Data provider for testGetCategories.
     *
     * @return array
     */
    public function getCategoriesDataProvider()
    {
        return [
            [2, 3, [6]],
            [null, 1, [6]],
        ];
    }
}
