<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SharedCatalog\Plugin\CatalogSearch\Model\Adapter\Mysql\Aggregation;

use Magento\Catalog\Model\ResourceModel\Product as ProductResourceModel;
use Magento\CatalogSearch\Model\Adapter\Mysql\Aggregation\DataProvider;
use Magento\Framework\Search\Request\BucketInterface;
use Magento\SharedCatalog\Api\StatusInfoInterface;
use Magento\Framework\DB\Select;
use Magento\SharedCatalog\Model\ResourceModel\ProductItem as ProductItemResourceModel;
use Magento\Store\Model\ScopeInterface;

/**
 * Class changes filter attribute logic for shared catalog.
 */
class DataProviderPlugin
{
    /**
     * @var StatusInfoInterface
     */
    private $statusInfo;

    /**
     * @var ProductItemResourceModel
     */
    private $productItem;

    /**
     * @var ProductResourceModel
     */
    private $product;

    /**
     * @param StatusInfoInterface $statusInfo
     * @param ProductItemResourceModel $productItem
     * @param ProductResourceModel $product
     */
    public function __construct(
        StatusInfoInterface $statusInfo,
        ProductItemResourceModel $productItem,
        ProductResourceModel $product
    ) {
        $this->statusInfo = $statusInfo;
        $this->productItem = $productItem;
        $this->product = $product;
    }

    /**
     * Join shared catalog item.
     *
     * @param DataProvider $subject
     * @param Select $select
     * @param BucketInterface $bucket
     * @return Select
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterGetDataSet(DataProvider $subject, Select $select, BucketInterface $bucket)
    {
        if (!$this->statusInfo->isActive(ScopeInterface::SCOPE_STORE, null) || $bucket->getName() == 'price_bucket') {
            return $select;
        }

        $innerSelect = $this->getSubSelect($select);
        if ($innerSelect !== false) {
            $innerSelect->joinInner(
                ['product_entity' => $this->product->getEntityTable()],
                'main_table.source_id  = product_entity.entity_id',
                ['sku']
            );
            $innerSelect->joinInner(
                ['shared_catalog_item' => $this->productItem->getMainTable()],
                'product_entity.sku  = shared_catalog_item.sku',
                []
            );
        }

        return $select;
    }

    /**
     * Get select statement.
     *
     * @param Select $select
     * @return Select|bool
     */
    private function getSubSelect(Select $select)
    {
        return $select->getPart('from')['main_table']['tableName'] ?: false;
    }
}
