<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\SharedCatalog\Model;

use Magento\CatalogPermissions\Helper\Data as PermissionsHelper;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\SharedCatalog\Model\SharedCatalogInvalidation;
use Magento\SharedCatalog\Model\CatalogPermissionManagement;
use Magento\Store\Model\StoreManagerInterface;
use Magento\SharedCatalog\Model\SharedCatalogAssignment;
use Magento\Store\Model\Store;
use Magento\Catalog\Model\Category;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;
use Magento\Customer\Api\Data\GroupInterface;
use Magento\SharedCatalog\Api\CategoryManagementInterface;

/**
 * Handle category management for shared catalog.
 */
class CategoryManagement implements CategoryManagementInterface
{
    /**
     * @var CategoryRepositoryInterface
     */
    private $categoryRepository;

    /**
     * @var SharedCatalogInvalidation
     */
    private $sharedCatalogInvalidation;

    /**
     * @var CatalogPermissionManagement
     */
    private $catalogPermissionManagement;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var SharedCatalogAssignment
     */
    private $sharedCatalogAssignment;

    /**
     * @var PermissionsHelper
     */
    private $permissionsHelper;

    /**
     * @param CategoryRepositoryInterface $categoryRepository
     * @param SharedCatalogInvalidation $sharedCatalogInvalidation
     * @param CatalogPermissionManagement $catalogPermissionManagement
     * @param StoreManagerInterface $storeManager
     * @param SharedCatalogAssignment $sharedCatalogAssignment
     * @param PermissionsHelper $permissionsHelper
     */
    public function __construct(
        CategoryRepositoryInterface $categoryRepository,
        SharedCatalogInvalidation $sharedCatalogInvalidation,
        CatalogPermissionManagement $catalogPermissionManagement,
        StoreManagerInterface $storeManager,
        SharedCatalogAssignment $sharedCatalogAssignment,
        PermissionsHelper $permissionsHelper
    ) {
        $this->categoryRepository = $categoryRepository;
        $this->sharedCatalogInvalidation = $sharedCatalogInvalidation;
        $this->catalogPermissionManagement = $catalogPermissionManagement;
        $this->storeManager = $storeManager;
        $this->sharedCatalogAssignment = $sharedCatalogAssignment;
        $this->permissionsHelper = $permissionsHelper;
    }

    /**
     * @inheritdoc
     */
    public function getCategories($id)
    {
        /** @var SharedCatalogInterface $sharedCatalog */
        $sharedCatalog = $this->sharedCatalogInvalidation->checkSharedCatalogExist($id);
        $storeId = $sharedCatalog->getStoreId();
        $allCategoriesIds = $this->getAllStoreCategoriesIds($storeId);
        if ($storeId === null) {
            $store = $this->storeManager->getGroup(Store::DEFAULT_STORE_ID);
        } else {
            $store = $this->storeManager->getGroup($storeId);
        }

        $websiteId = $store->getWebsiteId();
        $allowedCategoriesIds = $this->catalogPermissionManagement->getAllowedCategoriesIds(
            $id,
            $websiteId
        );
        $assignedCategoriesIds = array_intersect($allCategoriesIds, $allowedCategoriesIds);
        $assignedCategoriesIds = array_map(function ($value) {
            return (int)$value;
        }, $assignedCategoriesIds);

        return array_values($assignedCategoriesIds);
    }

    /**
     * Get all categories IDs for provided store by its ID.
     *
     * @param int $storeId
     * @return array
     * @throws NoSuchEntityException
     */
    private function getAllStoreCategoriesIds($storeId)
    {
        if ($storeId == Store::DEFAULT_STORE_ID) {
            $rootCategoryId = Category::TREE_ROOT_ID;
        } else {
            $store = $this->storeManager->getGroup($storeId);
            $rootCategoryId = $store->getRootCategoryId();
        }

        /** @var Category $rootCategory */
        $rootCategory = $this->categoryRepository->get($rootCategoryId);
        /** @var \Magento\Catalog\Model\ResourceModel\Category\Collection $categoryCollection */
        $categoryCollection = $rootCategory->getCollection();
        $categoryCollection->addPathsFilter($rootCategory->getPath() . '/');
        $categoriesIds = $categoryCollection->getAllIds();
        $categoriesIds[] = $rootCategory->getId();

        return $categoriesIds;
    }

    /**
     * @inheritdoc
     */
    public function assignCategories($id, array $categories)
    {
        /** @var SharedCatalogInterface $sharedCatalog */
        $sharedCatalog = $this->sharedCatalogInvalidation->checkSharedCatalogExist($id);
        $assignCategoriesIds = $this->retrieveCategoriesIds($categories);
        $customerGroups = $this->getSharedCatalogCustomerGroups($sharedCatalog);
        $this->catalogPermissionManagement->setAllowPermissions($assignCategoriesIds, $customerGroups);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function unassignCategories($id, array $categories)
    {
        /** @var SharedCatalogInterface $sharedCatalog */
        $sharedCatalog = $this->sharedCatalogInvalidation->checkSharedCatalogExist($id);
        $unassignCategoriesIds = $this->retrieveCategoriesIds($categories);
        $customerGroups = $this->getSharedCatalogCustomerGroups($sharedCatalog);
        $this->catalogPermissionManagement->setDenyPermissions($unassignCategoriesIds, $customerGroups);
        $this->sharedCatalogAssignment
            ->unassignProductsForCategories($id, $unassignCategoriesIds, $this->getCategories($id));

        return true;
    }

    /**
     * Retrieve categories Ids.
     *
     * @param \Magento\Catalog\Api\Data\CategoryInterface[] $categories
     * @return array
     * @throws NoSuchEntityException If some of the requested categories don't exist
     */
    private function retrieveCategoriesIds(array $categories)
    {
        $categoriesIds = [];
        foreach ($categories as $category) {
            $categoriesIds[] = $category->getId();
        }
        $allCategoriesIds = $this->getAllStoreCategoriesIds(Store::DEFAULT_STORE_ID);
        $nonexistentCategoriesIds = array_diff($categoriesIds, $allCategoriesIds);
        if (!empty($nonexistentCategoriesIds)) {
            throw new NoSuchEntityException(
                __(
                    'Requested categories don\'t exist: %categoriesIds.',
                    ['categoriesIds' => implode(', ', array_unique($nonexistentCategoriesIds))]
                )
            );
        }
        return $categoriesIds;
    }

    /**
     * Get list of shared catalog customer groups.
     *
     * @param SharedCatalogInterface $sharedCatalog
     * @return array
     */
    private function getSharedCatalogCustomerGroups(SharedCatalogInterface $sharedCatalog)
    {
        $customerGroups = [$sharedCatalog->getCustomerGroupId()];

        if ($sharedCatalog->getType() == SharedCatalogInterface::TYPE_PUBLIC) {
            $customerGroups[] = GroupInterface::NOT_LOGGED_IN_ID;
        }

        return $customerGroups;
    }
}
