<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\SharedCatalog\Observer;

use Magento\Company\Api\StatusServiceInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\DataObject;
use Magento\Framework\Event;
use Magento\Framework\Event\ObserverInterface;
use Magento\SharedCatalog\Model\Config as SharedCatalogModuleConfig;
use Magento\SharedCatalog\Model\Permissions\Config as PermissionsConfig;
use Magento\SharedCatalog\Model\Permissions\Synchronizer;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\SharedCatalog\Model\CatalogPermissionManagement;
use Magento\Framework\App\Config\ConfigResource\ConfigInterface as ConfigResource;

/**
 * Additional actions after saving data to system config.
 */
class UpdateConfig implements ObserverInterface
{
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var \Magento\Company\Api\StatusServiceInterface
     */
    private $companyStatusService;

    /**
     * @var \Magento\SharedCatalog\Model\Config
     */
    private $sharedCatalogModuleConfig;

    /**
     * @var PermissionsConfig
     */
    private $permissionsConfig;

    /**
     * @var \Magento\SharedCatalog\Model\CatalogPermissionManagement
     */
    private $catalogPermissionsManagement;

    /**
     * @var ConfigResource
     */
    private $configResource;

    /**
     * @var Synchronizer
     */
    private $permissionsSynchronizer;

    /**
     * @param StoreManagerInterface $storeManager
     * @param StatusServiceInterface $companyStatusService
     * @param SharedCatalogModuleConfig $sharedCatalogModuleConfig
     * @param PermissionsConfig $permissionsConfig
     * @param CatalogPermissionManagement $catalogPermissionsManagement
     * @param ConfigResource $configResource
     * @param Synchronizer $permissionsSynchronizer
     */
    public function __construct(
        StoreManagerInterface $storeManager,
        StatusServiceInterface $companyStatusService,
        SharedCatalogModuleConfig $sharedCatalogModuleConfig,
        PermissionsConfig $permissionsConfig,
        CatalogPermissionManagement $catalogPermissionsManagement,
        ConfigResource $configResource,
        Synchronizer $permissionsSynchronizer
    ) {
        $this->storeManager = $storeManager;
        $this->companyStatusService = $companyStatusService;
        $this->sharedCatalogModuleConfig = $sharedCatalogModuleConfig;
        $this->permissionsConfig = $permissionsConfig;
        $this->catalogPermissionsManagement = $catalogPermissionsManagement;
        $this->configResource = $configResource;
        $this->permissionsSynchronizer = $permissionsSynchronizer;
    }

    /**
     * {@inheritdoc}
     */
    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        $event = $observer->getEvent();
        $scopeData = $this->getEventScopeData($event);
        $scopeId = $scopeData->getScopeId() ?? null;

        $changedPaths = (array) $event->getData('changed_paths');
        $isCompanyChanged = \in_array('btob/website_configuration/company_active', $changedPaths);
        $isSharedCatalogChanged = \in_array(SharedCatalogModuleConfig::CONFIG_SHARED_CATALOG, $changedPaths);

        $isCompanyActive = $this->companyStatusService->isActive(
            $scopeData->getScopeType(),
            $scopeData->getScopeCode()
        );
        $isSharedCatalogActive = $this->sharedCatalogModuleConfig->isActive(
            $scopeData->getScopeType(),
            $scopeData->getScopeCode()
        );

        if ($isCompanyChanged && !$isCompanyActive && $isSharedCatalogActive) {
            $this->configResource->saveConfig(
                SharedCatalogModuleConfig::CONFIG_SHARED_CATALOG,
                false,
                $scopeData->getScopeType(),
                $scopeData->getScopeId()
            );

            $isSharedCatalogChanged = 1;
            $isSharedCatalogActive = 0;
        }
        if ($isSharedCatalogChanged) {
            if ($isSharedCatalogActive) {
                $this->permissionsConfig->enable($scopeId);
                $this->catalogPermissionsManagement->setPermissionsForAllCategories($scopeId);
            } else {
                $this->permissionsSynchronizer->removeCategoryPermissions($scopeId);
            }
        }
    }

    /**
     * Prepare scope data.
     *
     * @param Event $event
     * @return DataObject
     */
    private function getEventScopeData(Event $event)
    {
        $scopeData = new DataObject();
        $scopeType = $event->getWebsite()
            ? ScopeInterface::SCOPE_WEBSITES
            : ScopeConfigInterface::SCOPE_TYPE_DEFAULT;
        $scopeData->setScopeType($scopeType);

        $scopeData->setScopeCode('');
        $scopeData->setScopeId(null);
        if ($scopeType === ScopeInterface::SCOPE_WEBSITES) {
            $website = $this->storeManager->getWebsite($event->getWebsite());
            $scopeData->setScopeCode($website->getCode());
            $scopeData->setScopeId((int) $website->getId());
        }

        return $scopeData;
    }
}
