<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\SharedCatalog\Test\Unit\Model\ResourceModel\ProductItem\Price;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;

/**
 * Test for Magento/SharedCatalog/Model/ResourceModel/ProductItem/Price/Consumer class.
 */
class ConsumerTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var \Psr\Log\LoggerInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $loggerMock;

    /**
     * @var \Magento\Framework\EntityManager\EntityManager|\PHPUnit_Framework_MockObject_MockObject
     */
    private $entityManagerMock;

    /**
     * @var \Magento\Framework\Serialize\SerializerInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $serializerMock;

    /**
     * @var \Magento\Catalog\Api\TierPriceStorageInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $tierPriceStorage;

    /**
     * @var \Magento\Catalog\Api\Data\TierPriceInterfaceFactory|\PHPUnit_Framework_MockObject_MockObject
     */
    private $tierPriceFactory;

    /**
     * @var \Magento\AsynchronousOperations\Api\Data\OperationInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $operation;

    /**
     * @var \Magento\Catalog\Api\Data\TierPriceInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $tierPrice;

    /**
     * @var \Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price\Consumer
     */
    private $consumer;

    /**
     * {@inheritdoc}
     */
    protected function setUp()
    {
        $this->loggerMock = $this->getMockBuilder(\Psr\Log\LoggerInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->entityManagerMock = $this->getMockBuilder(\Magento\Framework\EntityManager\EntityManager::class)
            ->disableOriginalConstructor()
            ->setMethods(['save'])
            ->getMock();
        $this->serializerMock = $this->getMockBuilder(\Magento\Framework\Serialize\SerializerInterface::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->tierPriceStorage = $this->getMockBuilder(\Magento\Catalog\Api\TierPriceStorageInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->tierPriceFactory = $this->getMockBuilder(\Magento\Catalog\Api\Data\TierPriceInterfaceFactory::class)
            ->disableOriginalConstructor()
            ->setMethods(['create'])
            ->getMock();
        $this->operation = $this->getMockBuilder(\Magento\AsynchronousOperations\Api\Data\OperationInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->tierPrice = $this->getMockBuilder(\Magento\Catalog\Api\Data\TierPriceInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();

        $objectManagerHelper = new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);
        $this->consumer = $objectManagerHelper->getObject(
            \Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price\Consumer::class,
            [
                'logger' => $this->loggerMock,
                'entityManager' => $this->entityManagerMock,
                'serializer' => $this->serializerMock,
                'tierPriceStorage' => $this->tierPriceStorage,
                'tierPriceFactory' => $this->tierPriceFactory,
            ]
        );
    }

    /**
     * Test for \Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price\Consumer::processOperation.
     *
     * @param array $unserializedData
     * @return void
     * @dataProvider processOperationsDataProvider
     */
    public function testProcessOperationWhenExceptionOccurs(array $unserializedData)
    {
        $exceptionMessage = 'Exception message.';
        $exception = new \Exception($exceptionMessage);
        $params = ['SKU' => 'test_sku'];
        $message = 'Product with %SKU does not exist.';
        $serializedData = json_encode($unserializedData);
        $this->operation->expects($this->once())->method('getSerializedData')->willReturn($serializedData);
        $this->serializerMock->expects($this->once())->method('unserialize')->willReturn($unserializedData);
        $this->tierPriceFactory->expects($this->once())->method('create')->willReturn($this->tierPrice);
        $this->tierPrice->expects($this->atLeastOnce())->method('setPrice')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setPriceType')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setWebsiteId')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setSku')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setCustomerGroup')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setQuantity')->willReturnSelf();
        $priceUpdateResult = $this->getMockBuilder(\Magento\Catalog\Api\Data\PriceUpdateResultInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->tierPriceStorage->expects($this->once())
            ->method('delete')
            ->with([])
            ->willReturn([$priceUpdateResult]);
        $this->tierPriceStorage->expects($this->once())
            ->method('update')
            ->with([$this->tierPrice])
            ->willReturn([$priceUpdateResult]);
        $priceUpdateResult->expects($this->atLeastOnce())->method('getParameters')->willReturn($params);
        $priceUpdateResult->expects($this->atLeastOnce())->method('getMessage')->willReturn($message);
        $this->operation->expects($this->once())
            ->method('setStatus')
            ->with(OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED)
            ->willReturnSelf();
        $this->operation->expects($this->once())
            ->method('setResultMessage')
            ->with('Product with test_sku does not exist.')
            ->willReturnSelf();
        $operationList = $this->getMockBuilder(\Magento\AsynchronousOperations\Api\Data\OperationListInterface::class)
            ->disableOriginalConstructor()->getMock();
        $operationList->expects($this->atLeastOnce())->method('getItems')->willReturn([$this->operation]);
        $this->entityManagerMock->expects($this->once())
            ->method('save')->with($operationList)->willThrowException($exception);
        $this->loggerMock->expects($this->once())->method('critical')->with($exceptionMessage)->willReturnSelf();
        $this->consumer->processOperations($operationList);
    }

    /**
     * Test for processOperation().
     *
     * @param array $unserializedData
     * @return void
     * @dataProvider processOperationsDataProvider
     */
    public function testProcessOperations(array $unserializedData)
    {
        $params = ['SKU' => 'test_sku'];
        $message = 'Product with %SKU does not exist.';
        $serializedData = json_encode($unserializedData);
        $this->operation->expects($this->once())->method('getSerializedData')->willReturn($serializedData);
        $this->serializerMock->expects($this->once())->method('unserialize')->willReturn($unserializedData);
        $this->tierPriceFactory->expects($this->once())->method('create')->willReturn($this->tierPrice);
        $this->tierPrice->expects($this->atLeastOnce())->method('setWebsiteId')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setSku')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setCustomerGroup')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setQuantity')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setPrice')->willReturnSelf();
        $this->tierPrice->expects($this->atLeastOnce())->method('setPriceType')->willReturnSelf();
        $priceUpdateResult = $this->getMockBuilder(\Magento\Catalog\Api\Data\PriceUpdateResultInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->tierPriceStorage->expects($this->once())
            ->method('delete')
            ->with([])
            ->willReturn([$priceUpdateResult]);
        $this->tierPriceStorage->expects($this->once())
            ->method('update')
            ->with([$this->tierPrice])
            ->willReturn([$priceUpdateResult]);
        $priceUpdateResult->expects($this->atLeastOnce())->method('getParameters')->willReturn($params);
        $priceUpdateResult->expects($this->atLeastOnce())->method('getMessage')->willReturn($message);
        $this->operation->expects($this->once())
            ->method('setStatus')
            ->with(OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED)
            ->willReturnSelf();
        $this->operation->expects($this->once())
            ->method('setResultMessage')
            ->with('Product with test_sku does not exist.')
            ->willReturnSelf();
        $operationList = $this->getMockBuilder(\Magento\AsynchronousOperations\Api\Data\OperationListInterface::class)
            ->disableOriginalConstructor()
            ->setMethods(['getItems'])
            ->getMockForAbstractClass();
        $operationList->expects($this->atLeastOnce())->method('getItems')->willReturn([$this->operation]);
        $this->entityManagerMock->expects($this->once())->method('save')->with($operationList);

        $this->consumer->processOperations($operationList);
    }

    /**
     * Data provider for processOperations method.
     *
     * @return array
     */
    public function processOperationsDataProvider()
    {
        return [
            [
                [
                    'shared_catalog_id' => 1,
                    'entity_id' => 2,
                    'prices' => [
                        [
                            'qty' => 1,
                            'value_type' => 'percent',
                            'percentage_value' => 50,
                            'website_id' => 1,
                        ]
                    ],
                    'entity_link' => 'http://example.com',
                    'product_sku' => 'test_sku',
                    'customer_group' => 3,
                ]
            ],
            [
                [
                    'shared_catalog_id' => 1,
                    'entity_id' => 2,
                    'prices' => [
                        [
                            'qty' => 1,
                            'value_type' => 'fixed',
                            'price' => 20,
                            'website_id' => 1,
                        ]
                    ],
                    'entity_link' => 'http://example.com',
                    'product_sku' => 'test_sku',
                    'customer_group' => 3,
                ]
            ],
        ];
    }
}
