<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\SharedCatalog\Model\ResourceModel\ProductItem\Price;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\EntityManager\EntityManager;

/**
 * Consumer for tier prices update queue.
 */
class Consumer
{
    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    /**
     * @var EntityManager
     */
    private $entityManager;

    /**
     * @var \Magento\Framework\Serialize\SerializerInterface
     */
    private $serializer;

    /**
     * @var \Magento\Catalog\Api\TierPriceStorageInterface
     */
    private $tierPriceStorage;

    /**
     * @var \Magento\Catalog\Api\Data\TierPriceInterfaceFactory
     */
    private $tierPriceFactory;

    /**
     * @param \Psr\Log\LoggerInterface $logger
     * @param EntityManager $entityManager
     * @param \Magento\Framework\Serialize\SerializerInterface $serializer
     * @param \Magento\Catalog\Api\TierPriceStorageInterface $tierPriceStorage
     * @param \Magento\Catalog\Api\Data\TierPriceInterfaceFactory $tierPriceFactory
     */
    public function __construct(
        \Psr\Log\LoggerInterface $logger,
        EntityManager $entityManager,
        \Magento\Framework\Serialize\SerializerInterface $serializer,
        \Magento\Catalog\Api\TierPriceStorageInterface $tierPriceStorage,
        \Magento\Catalog\Api\Data\TierPriceInterfaceFactory $tierPriceFactory
    ) {
        $this->logger = $logger;
        $this->entityManager = $entityManager;
        $this->serializer = $serializer;
        $this->tierPriceStorage = $tierPriceStorage;
        $this->tierPriceFactory = $tierPriceFactory;
    }

    /**
     * Processing batch of operations for update tier prices.
     *
     * @param \Magento\AsynchronousOperations\Api\Data\OperationListInterface $operationList
     * @return void
     */
    public function processOperations(\Magento\AsynchronousOperations\Api\Data\OperationListInterface $operationList)
    {
        $pricesUpdateDto = [];
        $pricesDeleteDto = [];
        $operationSkus = [];
        foreach ($operationList->getItems() as $index => $operation) {
            $serializedData = $operation->getSerializedData();
            $unserializedData = $this->serializer->unserialize($serializedData);
            $operationSkus[$index] = $unserializedData['product_sku'];
            $pricesUpdateDto = array_merge($pricesUpdateDto, $this->createPricesUpdate($unserializedData));
            $pricesDeleteDto = array_merge($pricesDeleteDto, $this->createPricesDelete($unserializedData));
        }

        $failedOperations = [];
        $failedDeleteItems = $this->tierPriceStorage->delete($pricesDeleteDto);
        $failedUpdateItems = $this->tierPriceStorage->update($pricesUpdateDto);
        $failedItems = array_merge($failedDeleteItems, $failedUpdateItems);
        foreach ($failedItems as $failedItem) {
            if (isset($failedItem->getParameters()['SKU'])) {
                $failedOperations[$failedItem->getParameters()['SKU']] = $this->prepareErrorMessage($failedItem);
            }
        }

        try {
            foreach ($operationList->getItems() as $index => $operation) {
                // save operation data and status
                if (isset($failedOperations[$operationSkus[$index]])) {
                    $operation->setStatus(OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED);
                    $operation->setResultMessage($failedOperations[$operationSkus[$index]]);
                } else {
                    $operation->setStatus(OperationInterface::STATUS_TYPE_COMPLETE);
                    $operation->setResultMessage(null);
                }
            }
            $this->entityManager->save($operationList);
        } catch (\Exception $exception) {
            // prevent consumer from failing, silently log exception
            $this->logger->critical($exception->getMessage());
        }
    }

    /**
     * Get formatted error message by replacing placeholders in it with values.
     *
     * @param \Magento\Catalog\Api\Data\PriceUpdateResultInterface $result
     * @return string
     */
    private function prepareErrorMessage(\Magento\Catalog\Api\Data\PriceUpdateResultInterface $result)
    {
        $message = $result->getMessage();
        foreach ($result->getParameters() as $placeholder => $value) {
            $message = str_replace('%' . $placeholder, $value, $message);
        }
        return $message;
    }

    /**
     * Create tier prices DTO and populate it with data from the operation for update.
     *
     * @param array $operationData
     * @return \Magento\Catalog\Api\Data\TierPriceInterface[]
     */
    private function createPricesUpdate(array $operationData)
    {
        $pricesDto = [];
        foreach ($operationData['prices'] as $price) {
            if (!empty($price['is_deleted'])) {
                continue;
            }
            $pricesDto[] = $this->createPrice(
                $operationData['product_sku'],
                $operationData['customer_group'],
                $price
            );
        }
        return $pricesDto;
    }

    /**
     * Create tier prices DTO and populate it with data from the operation for delete.
     *
     * @param array $operationData
     * @return \Magento\Catalog\Api\Data\TierPriceInterface[]
     */
    private function createPricesDelete(array $operationData)
    {
        $pricesDto = [];
        foreach ($operationData['prices'] as $price) {
            if (empty($price['is_deleted'])) {
                continue;
            }
            $pricesDto[] = $this->createPrice(
                $operationData['product_sku'],
                $operationData['customer_group'],
                $price
            );
        }
        return $pricesDto;
    }

    /**
     * Create tier prices DTO with $sku, $group and price type and value form $price.
     *
     * @param string $sku
     * @param string $group
     * @param array $price
     * @return \Magento\Catalog\Api\Data\TierPriceInterface
     */
    private function createPrice($sku, $group, array $price)
    {
        /** @var \Magento\Catalog\Api\Data\TierPriceInterface $priceDto */
        $priceDto = $this->tierPriceFactory->create();
        $priceDto
            ->setWebsiteId($price['website_id'])
            ->setSku($sku)
            ->setCustomerGroup($group)
            ->setQuantity($price['qty']);
        if ($price['value_type'] == \Magento\Catalog\Api\Data\TierPriceInterface::PRICE_TYPE_FIXED) {
            $priceDto
                ->setPrice($price['price'])
                ->setPriceType(\Magento\Catalog\Api\Data\TierPriceInterface::PRICE_TYPE_FIXED);
        } else {
            $priceDto
                ->setPrice($price['percentage_value'])
                ->setPriceType(\Magento\Catalog\Api\Data\TierPriceInterface::PRICE_TYPE_DISCOUNT);
        }
        return $priceDto;
    }
}
