<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\SharedCatalog\Test\Unit\Model;

/**
 * SaveHandler unit test.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class SaveHandlerTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var \Magento\SharedCatalog\Model\ResourceModel\SharedCatalog|\PHPUnit_Framework_MockObject_MockObject
     */
    private $sharedCatalogResource;

    /**
     * @var \Magento\SharedCatalog\Api\ProductItemManagementInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $sharedCatalogProductItemManagement;

    /**
     * @var \Magento\SharedCatalog\Model\CustomerGroupManagement|\PHPUnit_Framework_MockObject_MockObject
     */
    private $customerGroupManagement;

    /**
     * @var \Magento\SharedCatalog\Api\SharedCatalogManagementInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $sharedCatalogManagement;

    /**
     * @var \Magento\SharedCatalog\Model\CatalogPermissionManagement|\PHPUnit_Framework_MockObject_MockObject
     */
    private $catalogPermissionManagement;

    /**
     * @var \Magento\SharedCatalog\Model\SharedCatalogValidator|\PHPUnit_Framework_MockObject_MockObject
     */
    private $validator;

    /**
     * @var \Magento\SharedCatalog\Api\CompanyManagementInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $sharedCatalogCompanyManagement;

    /**
     * @var \Magento\Authorization\Model\UserContextInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $userContext;

    /**
     * @var \Magento\SharedCatalog\Api\CategoryManagementInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    private $categoryManagement;

    /**
     * @var \Magento\SharedCatalog\Model\SaveHandler
     */
    private $saveHandler;

    /**
     * Set up.
     *
     * @return void
     */
    protected function setUp()
    {
        $this->sharedCatalogResource = $this
            ->getMockBuilder(\Magento\SharedCatalog\Model\ResourceModel\SharedCatalog::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->sharedCatalogProductItemManagement = $this
            ->getMockBuilder(\Magento\SharedCatalog\Api\ProductItemManagementInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->customerGroupManagement = $this
            ->getMockBuilder(\Magento\SharedCatalog\Model\CustomerGroupManagement::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->sharedCatalogManagement = $this
            ->getMockBuilder(\Magento\SharedCatalog\Api\SharedCatalogManagementInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->catalogPermissionManagement = $this
            ->getMockBuilder(\Magento\SharedCatalog\Model\CatalogPermissionManagement::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->validator = $this
            ->getMockBuilder(\Magento\SharedCatalog\Model\SharedCatalogValidator::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->sharedCatalogCompanyManagement = $this
            ->getMockBuilder(\Magento\SharedCatalog\Api\CompanyManagementInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->userContext = $this
            ->getMockBuilder(\Magento\Authorization\Model\UserContextInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $this->categoryManagement = $this
            ->getMockBuilder(\Magento\SharedCatalog\Api\CategoryManagementInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $objectManager = new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);
        $this->saveHandler = $objectManager->getObject(
            \Magento\SharedCatalog\Model\SaveHandler::class,
            [
                'sharedCatalogResource' => $this->sharedCatalogResource,
                'sharedCatalogProductItemManagement' => $this->sharedCatalogProductItemManagement,
                'customerGroupManagement' => $this->customerGroupManagement,
                'sharedCatalogManagement' => $this->sharedCatalogManagement,
                'catalogPermissionManagement' => $this->catalogPermissionManagement,
                'validator' => $this->validator,
                'sharedCatalogCompanyManagement' => $this->sharedCatalogCompanyManagement,
                'userContext' => $this->userContext,
                'categoryManagement' => $this->categoryManagement,
            ]
        );
    }

    /**
     * Test execute.
     *
     * @return void
     */
    public function testExecute()
    {
        $storeCategoriesIds = [1];
        $sharedCatalog = $this->getMockBuilder(\Magento\SharedCatalog\Model\SharedCatalog::class)
            ->disableOriginalConstructor()
            ->getMock();
        $sharedCatalog->expects($this->atLeastOnce())->method('getCustomerGroupId')->willReturn(null);
        $sharedCatalog->expects($this->once())->method('setCreatedBy')->willReturnSelf();
        $sharedCatalog->expects($this->once())->method('setCustomerGroupId')->willReturnSelf();
        $sharedCatalog->expects($this->atLeastOnce())->method('getType')
            ->willReturn(\Magento\SharedCatalog\Api\Data\SharedCatalogInterface::TYPE_PUBLIC);
        $this->validator->expects($this->once())->method('validate')->with($sharedCatalog);
        $this->validator->expects($this->once())->method('isCatalogPublicTypeDuplicated')->with($sharedCatalog)
            ->willReturn(false);
        $publicCatalog = $this->getMockBuilder(\Magento\SharedCatalog\Model\SharedCatalog::class)
            ->disableOriginalConstructor()
            ->getMock();
        $publicCatalog->expects($this->never())->method('setType')
            ->with(\Magento\SharedCatalog\Api\Data\SharedCatalogInterface::TYPE_CUSTOM)->willReturnSelf();
        $this->sharedCatalogManagement->expects($this->never())->method('getPublicCatalog')->willReturn($publicCatalog);
        $this->sharedCatalogProductItemManagement->expects($this->never())->method('deletePricesForPublicCatalog')
            ->with($publicCatalog);
        $this->userContext->expects($this->once())->method('getUserType')
            ->willReturn(\Magento\Authorization\Model\UserContextInterface::USER_TYPE_ADMIN);
        $this->userContext->expects($this->once())->method('getUserId')->willReturn(1);
        $customerGroup = $this->getMockBuilder(\Magento\Customer\Api\Data\GroupInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $customerGroup->expects($this->once())->method('getId')->willReturn(1);
        $this->customerGroupManagement->expects($this->once())->method('createCustomerGroupForSharedCatalog')
            ->with($sharedCatalog)->willReturn($customerGroup);
        $this->customerGroupManagement->expects($this->atLeastOnce())->method('getGroupIds')->with(true)
            ->willReturn([1]);
        $this->catalogPermissionManagement->expects($this->atLeastOnce())->method('reassignForRootCategories');
        $this->customerGroupManagement->expects($this->never())->method('setDefaultCustomerGroup')
            ->with($sharedCatalog);
        $this->customerGroupManagement->expects($this->once())->method('updateCustomerGroup');
        $sharedCatalog->expects($this->once())->method('getId')->willReturn(false);
        $this->catalogPermissionManagement->expects($this->once())->method('retrieveCategoriesIds')
            ->with(null)->willReturn($storeCategoriesIds);
        $this->catalogPermissionManagement->expects($this->once())->method('setDenyPermissions')
            ->with($storeCategoriesIds, [null]);
        $this->sharedCatalogProductItemManagement->expects($this->once())->method('addPricesForPublicCatalog');
        $this->sharedCatalogCompanyManagement->expects($this->never())->method('unassignAllCompanies');

        $this->sharedCatalogResource->expects($this->once())->method('save')->with($sharedCatalog);
        $this->validator->expects($this->once())->method('isDirectChangeToCustom')->with($sharedCatalog);
        $this->sharedCatalogManagement->expects($this->once())->method('isPublicCatalogExist')
            ->willReturn(true);

        $this->assertInstanceOf(
            \Magento\SharedCatalog\Api\Data\SharedCatalogInterface::class,
            $this->saveHandler->execute($sharedCatalog)
        );
    }

    /**
     * Test execute when public catalog is duplicated.
     *
     * @return void
     */
    public function testExecutePublicCatalogDuplicated()
    {
        $sharedCatalog = $this->getMockBuilder(\Magento\SharedCatalog\Model\SharedCatalog::class)
            ->disableOriginalConstructor()
            ->getMock();
        $sharedCatalog->expects($this->atLeastOnce())->method('getCustomerGroupId')->willReturn(null);
        $sharedCatalog->expects($this->once())->method('setCreatedBy')->willReturnSelf();
        $sharedCatalog->expects($this->once())->method('setCustomerGroupId')->willReturnSelf();
        $sharedCatalog->expects($this->once())->method('getType')->willReturn(null);
        $sharedCatalog->expects($this->once())->method('setType')
            ->with(\Magento\SharedCatalog\Api\Data\SharedCatalogInterface::TYPE_CUSTOM)->willReturnSelf();
        $this->validator->expects($this->once())->method('validate')->with($sharedCatalog);
        $this->validator->expects($this->once())->method('isCatalogPublicTypeDuplicated')->with($sharedCatalog)
            ->willReturn(true);
        $publicCatalog = $this->getMockBuilder(\Magento\SharedCatalog\Model\SharedCatalog::class)
            ->disableOriginalConstructor()
            ->getMock();
        $publicCatalog->expects($this->once())->method('setType')
            ->with(\Magento\SharedCatalog\Api\Data\SharedCatalogInterface::TYPE_CUSTOM)->willReturnSelf();
        $this->sharedCatalogManagement->expects($this->once())->method('getPublicCatalog')->willReturn($publicCatalog);
        $this->sharedCatalogProductItemManagement->expects($this->once())->method('deletePricesForPublicCatalog');
        $this->userContext->expects($this->once())->method('getUserType')
            ->willReturn(\Magento\Authorization\Model\UserContextInterface::USER_TYPE_ADMIN);
        $this->userContext->expects($this->once())->method('getUserId')->willReturn(1);
        $customerGroup = $this->getMockBuilder(\Magento\Customer\Api\Data\GroupInterface::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $customerGroup->expects($this->once())->method('getId')->willReturn(1);
        $this->customerGroupManagement->expects($this->once())->method('createCustomerGroupForSharedCatalog')
            ->with($sharedCatalog)->willReturn($customerGroup);
        $this->customerGroupManagement->expects($this->once())->method('getGroupIds')->with(true)->willReturn([1]);
        $this->catalogPermissionManagement->expects($this->once())->method('reassignForRootCategories');

        $this->sharedCatalogResource->expects($this->exactly(2))->method('save')->with($sharedCatalog)
            ->willReturn($sharedCatalog);
        $this->customerGroupManagement->expects($this->once())->method('setDefaultCustomerGroup')->with($sharedCatalog);
        $this->sharedCatalogProductItemManagement->expects($this->exactly(1))->method('addPricesForPublicCatalog');
        $this->sharedCatalogCompanyManagement->expects($this->once())->method('unassignAllCompanies');
        $this->categoryManagement->expects($this->atLeastOnce())->method('getCategories')->willReturn([1]);
        $this->catalogPermissionManagement->expects($this->once())->method('setDenyPermissions');
        $this->catalogPermissionManagement->expects($this->once())->method('setAllowPermissions');

        $this->assertInstanceOf(
            \Magento\SharedCatalog\Api\Data\SharedCatalogInterface::class,
            $this->saveHandler->execute($sharedCatalog)
        );
    }

    /**
     * Test execute with exception.
     *
     * @return void
     * @expectedException \Exception
     */
    public function testExecuteWithException()
    {
        $exception = new \Exception();
        $this->validator->expects($this->once())->method('validate')->willThrowException($exception);
        $sharedCatalog = $this->getMockBuilder(\Magento\SharedCatalog\Model\SharedCatalog::class)
            ->disableOriginalConstructor()
            ->getMock();
        $sharedCatalog->expects($this->atLeastOnce())->method('getCustomerGroupId')->willReturn(null);

        $this->saveHandler->execute($sharedCatalog);
    }
}
