<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\SharedCatalog\Model;

use Magento\SharedCatalog\Api\Data\SharedCatalogInterface;

/**
 * Handler for shared catalog save.
 */
class SaveHandler
{
    /**
     * @var \Magento\SharedCatalog\Model\ResourceModel\SharedCatalog
     */
    private $sharedCatalogResource;

    /**
     * @var \Magento\SharedCatalog\Api\ProductItemManagementInterface
     */
    private $sharedCatalogProductItemManagement;

    /**
     * @var \Magento\SharedCatalog\Model\CustomerGroupManagement
     */
    private $customerGroupManagement;

    /**
     * @var \Magento\SharedCatalog\Api\SharedCatalogManagementInterface
     */
    private $sharedCatalogManagement;

    /**
     * @var \Magento\SharedCatalog\Model\SharedCatalogValidator
     */
    private $validator;

    /**
     * @var \Magento\SharedCatalog\Model\CatalogPermissionManagement
     */
    private $catalogPermissionManagement;

    /**
     * @var \Magento\SharedCatalog\Api\CompanyManagementInterface
     */
    private $sharedCatalogCompanyManagement;

    /**
     * @var \Magento\Authorization\Model\UserContextInterface
     */
    private $userContext;

    /**
     * @var \Magento\SharedCatalog\Api\CategoryManagementInterface
     */
    private $categoryManagement;

    /**
     * @param ResourceModel\SharedCatalog $sharedCatalogResource
     * @param \Magento\SharedCatalog\Api\ProductItemManagementInterface $sharedCatalogProductItemManagement
     * @param \Magento\SharedCatalog\Model\CustomerGroupManagement $customerGroupManagement
     * @param \Magento\SharedCatalog\Api\SharedCatalogManagementInterface $sharedCatalogManagement
     * @param \Magento\SharedCatalog\Model\CatalogPermissionManagement $catalogPermissionManagement
     * @param \Magento\SharedCatalog\Api\CompanyManagementInterface $sharedCatalogCompanyManagement
     * @param \Magento\SharedCatalog\Model\SharedCatalogValidator $validator
     * @param \Magento\Authorization\Model\UserContextInterface $userContext
     * @param \Magento\SharedCatalog\Api\CategoryManagementInterface $categoryManagement
     */
    public function __construct(
        \Magento\SharedCatalog\Model\ResourceModel\SharedCatalog $sharedCatalogResource,
        \Magento\SharedCatalog\Api\ProductItemManagementInterface $sharedCatalogProductItemManagement,
        \Magento\SharedCatalog\Model\CustomerGroupManagement $customerGroupManagement,
        \Magento\SharedCatalog\Api\SharedCatalogManagementInterface $sharedCatalogManagement,
        \Magento\SharedCatalog\Model\CatalogPermissionManagement $catalogPermissionManagement,
        \Magento\SharedCatalog\Model\SharedCatalogValidator $validator,
        \Magento\SharedCatalog\Api\CompanyManagementInterface $sharedCatalogCompanyManagement,
        \Magento\Authorization\Model\UserContextInterface $userContext,
        \Magento\SharedCatalog\Api\CategoryManagementInterface $categoryManagement
    ) {
        $this->sharedCatalogResource = $sharedCatalogResource;
        $this->sharedCatalogProductItemManagement = $sharedCatalogProductItemManagement;
        $this->customerGroupManagement = $customerGroupManagement;
        $this->sharedCatalogManagement = $sharedCatalogManagement;
        $this->catalogPermissionManagement = $catalogPermissionManagement;
        $this->sharedCatalogCompanyManagement = $sharedCatalogCompanyManagement;
        $this->validator = $validator;
        $this->userContext = $userContext;
        $this->categoryManagement = $categoryManagement;
    }

    /**
     * Shared Catalog saving.
     *
     * Saving shared catalog. If it is a new shared catalog, creating customer group. If it is an existing shared
     * catalog and the shared catalog name is changing, updating related customer group name. If a shared catalog type
     * is being changed to public, reassigning all companies from the current public shared catalog to the new public
     * shared catalog.
     *
     * @param SharedCatalogInterface $sharedCatalog
     * @return SharedCatalogInterface
     * @throws \Exception
     */
    public function execute(SharedCatalogInterface $sharedCatalog)
    {
        $originalSharedCatalog = clone $sharedCatalog;
        try {
            $this->validator->validate($sharedCatalog);

            if ($this->validator->isCatalogPublicTypeDuplicated($sharedCatalog)) {
                $publicCatalog = $this->sharedCatalogManagement->getPublicCatalog();
                $publicCatalog->setType(SharedCatalogInterface::TYPE_CUSTOM);
                $this->sharedCatalogProductItemManagement->deletePricesForPublicCatalog();
                $this->sharedCatalogResource->save($publicCatalog);
                $sharedCatalog = $this->prepareCatalog($sharedCatalog);
                $this->sharedCatalogResource->save($sharedCatalog);
                $this->customerGroupManagement->updateCustomerGroup($sharedCatalog);
                $this->customerGroupManagement->setDefaultCustomerGroup($sharedCatalog);
                $this->sharedCatalogProductItemManagement->addPricesForPublicCatalog();
                $this->sharedCatalogCompanyManagement->unassignAllCompanies($publicCatalog->getId());
                $this->catalogPermissionManagement->setDenyPermissions(
                    $this->categoryManagement->getCategories($publicCatalog->getId()),
                    [\Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID]
                );
                $this->catalogPermissionManagement->setAllowPermissions(
                    $this->categoryManagement->getCategories($sharedCatalog->getId()),
                    [\Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID]
                );
                return $sharedCatalog;
            }
            $this->validator->isDirectChangeToCustom($sharedCatalog);
            $isNotFirstCatalogToCreate = $this->sharedCatalogManagement->isPublicCatalogExist()
                && ($sharedCatalog->getType() == SharedCatalogInterface::TYPE_PUBLIC);
            $sharedCatalog = $this->prepareCatalog($sharedCatalog);
            $this->sharedCatalogResource->save($sharedCatalog);
            $this->customerGroupManagement->updateCustomerGroup($sharedCatalog);
            if (!$originalSharedCatalog->getId()) {
                $this->setDenyPermissionsForNewSharedCatalog($sharedCatalog);
            }
            if ($isNotFirstCatalogToCreate) {
                $customerGroupIds = $this->customerGroupManagement->getGroupIds(true);
                $this->catalogPermissionManagement->reassignForRootCategories($customerGroupIds);
                $this->sharedCatalogProductItemManagement->addPricesForPublicCatalog();
            }
        } catch (\Exception $e) {
            $this->rollback($sharedCatalog, $originalSharedCatalog);
            throw $e;
        }
        return $sharedCatalog;
    }

    /**
     * Set deny permissions for all shared catalog categories if the catalog is new.
     *
     * @param SharedCatalogInterface $sharedCatalog
     * @return void
     */
    private function setDenyPermissionsForNewSharedCatalog(SharedCatalogInterface $sharedCatalog)
    {
        $categoryIds = $this->catalogPermissionManagement->retrieveCategoriesIds(null);
        $this->catalogPermissionManagement->setDenyPermissions(
            $categoryIds,
            [$sharedCatalog->getCustomerGroupId()]
        );
    }

    /**
     * Prepare shared catalog data before save.
     *
     * @param SharedCatalogInterface $sharedCatalog
     * @return SharedCatalogInterface
     */
    private function prepareCatalog(SharedCatalogInterface $sharedCatalog)
    {
        if ($sharedCatalog->getCustomerGroupId()) {
            return $sharedCatalog;
        }
        $sharedCatalog->setCreatedBy($this->getCurrentUserId());
        $customerGroup = $this->customerGroupManagement->createCustomerGroupForSharedCatalog($sharedCatalog);
        $sharedCatalog->setCustomerGroupId($customerGroup->getId());
        $customerGroupIds = $this->customerGroupManagement->getGroupIds(true);
        $this->catalogPermissionManagement->reassignForRootCategories($customerGroupIds, $sharedCatalog);
        if ($sharedCatalog->getType() === null) {
            $sharedCatalog->setType(SharedCatalogInterface::TYPE_CUSTOM);
        }

        return $sharedCatalog;
    }

    /**
     * Rollback saving transaction.
     *
     * @param SharedCatalogInterface $sharedCatalog
     * @param SharedCatalogInterface $originalSharedCatalog
     * @return void
     */
    private function rollback(
        SharedCatalogInterface $sharedCatalog,
        SharedCatalogInterface $originalSharedCatalog
    ) {
        if (!$originalSharedCatalog->getCustomerGroupId() && $sharedCatalog->getCustomerGroupId()) {
            $this->customerGroupManagement->deleteCustomerGroupById($sharedCatalog);
            $sharedCatalog->setCustomerGroupId(null);
        }
    }

    /**
     * Identify current user ID.
     *
     * @return int|null
     */
    private function getCurrentUserId()
    {
        if ($this->userContext->getUserType() != \Magento\Authorization\Model\UserContextInterface::USER_TYPE_ADMIN) {
            return null;
        }

        return $this->userContext->getUserId();
    }
}
