<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\SharedCatalog\Model;

/**
 * Management Customer Group for SharedCatalog.
 */
class CustomerGroupManagement
{
    /**
     * @var \Magento\Customer\Model\ResourceModel\Group\CollectionFactory
     */
    private $groupCollectionFactory;

    /**
     * @var bool|null
     */
    private $masterCatalog = null;

    /**
     * @var \Magento\Customer\Api\CustomerGroupConfigInterface
     */
    private $customerGroupConfig;

    /**
     * @var \Magento\Customer\Api\Data\GroupInterfaceFactory
     */
    private $groupFactory;

    /**
     * @var \Magento\Customer\Api\GroupRepositoryInterface
     */
    private $groupRepository;

    /**
     * @var \Magento\Catalog\Api\CategoryRepositoryInterface
     */
    private $categoryRepository;

    /**
     * CustomerGroupManagement constructor.
     *
     * @param \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $groupCollectionFactory
     * @param \Magento\Customer\Api\CustomerGroupConfigInterface $customerGroupConfig
     * @param \Magento\Customer\Api\Data\GroupInterfaceFactory $groupFactory
     * @param \Magento\Customer\Api\GroupRepositoryInterface $groupRepository
     * @param \Magento\Catalog\Api\CategoryRepositoryInterface $categoryRepository
     */
    public function __construct(
        \Magento\Customer\Model\ResourceModel\Group\CollectionFactory $groupCollectionFactory,
        \Magento\Customer\Api\CustomerGroupConfigInterface $customerGroupConfig,
        \Magento\Customer\Api\Data\GroupInterfaceFactory $groupFactory,
        \Magento\Customer\Api\GroupRepositoryInterface $groupRepository,
        \Magento\Catalog\Api\CategoryRepositoryInterface $categoryRepository
    ) {
        $this->groupCollectionFactory = $groupCollectionFactory;
        $this->customerGroupConfig = $customerGroupConfig;
        $this->groupFactory = $groupFactory;
        $this->groupRepository = $groupRepository;
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * Check if master catalog should be displayed for customer group.
     *
     * @param int $customerGroupId
     * @return bool
     */
    public function isMasterCatalogAvailable($customerGroupId)
    {
        if ($this->masterCatalog === null) {
            $this->masterCatalog = in_array($customerGroupId, $this->getGroupIds());
        }
        return $this->masterCatalog;
    }

    /**
     * Get customer groups that are not linked to any shared catalog.
     *
     * @param bool $isSharedCatalog [optional]
     * @return array
     */
    public function getGroupIds($isSharedCatalog = false)
    {
        $collection = $this->groupCollectionFactory->create();
        $collection->getSelect()->joinLeft(
            ['shared_catalog' => $collection->getTable('shared_catalog')],
            'main_table.customer_group_id = shared_catalog.customer_group_id',
            ['shared_catalog_id' => 'shared_catalog.entity_id']
        );
        $collection->getSelect()->where(
            ($isSharedCatalog ? 'NOT' : '')
            . '(shared_catalog.entity_id IS NULL AND main_table.customer_group_id != ?)',
            \Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID
        );
        return $collection->getColumnValues('customer_group_id');
    }

    /**
     * Set default customer group.
     *
     * @param \Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog
     * @throws \UnexpectedValueException
     * @throws \Exception
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Magento\Framework\Exception\LocalizedException
     * @return void
     */
    public function setDefaultCustomerGroup(\Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog)
    {
        $this->customerGroupConfig->setDefaultCustomerGroup($sharedCatalog->getCustomerGroupId());
    }

    /**
     * Create customer group for SharedCatalog.
     *
     * @param \Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog
     * @return \Magento\Customer\Api\Data\GroupInterface
     */
    public function createCustomerGroupForSharedCatalog(
        \Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog
    ) {
        /** @var \Magento\Customer\Api\Data\GroupInterface $customerGroup */
        $customerGroup = $this->groupFactory->create();
        $customerGroup->setCode($sharedCatalog->getName());
        if ($sharedCatalog->getTaxClassId()) {
            $customerGroup->setTaxClassId($sharedCatalog->getTaxClassId());
        }
        return $this->groupRepository->save($customerGroup);
    }

    /**
     * Delete customer group by ID.
     *
     * @param \Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog
     * @return bool true on success
     */
    public function deleteCustomerGroupById(\Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog)
    {
        return $this->groupRepository->deleteById($sharedCatalog->getCustomerGroupId());
    }

    /**
     * Update customer group code and tax class id.
     *
     * @param \Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog
     * @return bool
     */
    public function updateCustomerGroup(\Magento\SharedCatalog\Api\Data\SharedCatalogInterface $sharedCatalog)
    {
        $customerGroup = $this->groupRepository->getById($sharedCatalog->getCustomerGroupId());
        $changeCustomerGroupTaxClassIdResult = $this->changeTaxClassId($customerGroup, $sharedCatalog->getTaxClassId());
        $changeCustomerGroupCodeResult = $this->changeCustomerGroupCode($customerGroup, $sharedCatalog->getName());

        if ($changeCustomerGroupTaxClassIdResult || $changeCustomerGroupCodeResult) {
            $this->groupRepository->save($customerGroup);
            return true;
        }

        return false;
    }

    /**
     * Set customer group tax class id if new tax class id differs from the initial one.
     *
     * @param \Magento\Customer\Api\Data\GroupInterface $customerGroup
     * @param int $taxClassId
     * @return bool
     */
    private function changeTaxClassId(\Magento\Customer\Api\Data\GroupInterface $customerGroup, $taxClassId)
    {
        if ($customerGroup && $customerGroup->getTaxClassId() != $taxClassId) {
            $customerGroup->setTaxClassId($taxClassId);
            return true;
        }

        return false;
    }

    /**
     * Set customer group code if new code differs from the initial one and customer group is Not Logged In.
     *
     * @param \Magento\Customer\Api\Data\GroupInterface $customerGroup
     * @param string $customerGroupCode
     * @return bool
     */
    private function changeCustomerGroupCode(
        \Magento\Customer\Api\Data\GroupInterface $customerGroup,
        $customerGroupCode
    ) {
        if ($customerGroup && $customerGroup->getId() != \Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID
            && $customerGroup->getCode() != $customerGroupCode
        ) {
            $customerGroup->setCode($customerGroupCode);
            return true;
        }

        return false;
    }
}
