<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\ServicesId\Model;

use Magento\Framework\App\Cache\Type\Config as CacheConfig;
use Magento\Framework\App\Cache\TypeListInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;

/**
 * @inheritdoc
 */
class ServicesConfig implements ServicesConfigInterface
{
    /**
     * Config path values for Services Id
     */
    public const CONFIG_PATH_PROJECT_ID = 'services_connector/services_id/project_id';
    public const CONFIG_PATH_PROJECT_NAME = 'services_connector/services_id/project_name';
    public const CONFIG_PATH_ENVIRONMENT_ID = 'services_connector/services_id/environment_id';
    public const CONFIG_PATH_ENVIRONMENT_NAME = 'services_connector/services_id/environment_name';
    public const CONFIG_PATH_ENVIRONMENT_TYPE = 'services_connector/services_id/environment';
    public const CONFIG_PATH_IMS_ORGANIZATION_ID = 'services_connector/services_id/ims_organization_id';
    public const CONFIG_PATH_REGISTRY_API_PATH = 'services_connector/services_id/registry_api_path';

    /**
     * Config path values for Services Connector
     */
    public const CONFIG_PATH_SERVICES_CONNECTOR_ENVIRONMENT = 'magento_saas/environment';
    public const CONFIG_PATH_SERVICES_CONNECTOR_API_KEY =
        'services_connector/services_connector_integration/{env}_api_key';
    public const CONFIG_PATH_SERVICES_CONNECTOR_PRIVATE_KEY =
        'services_connector/services_connector_integration/{env}_private_key';
    public const CONFIG_PATH_SERVICES_CONNECTOR_SANDBOX_API_KEY =
        'services_connector/services_connector_integration/sandbox_api_key';
    public const CONFIG_PATH_SERVICES_CONNECTOR_SANDBOX_PRIVATE_KEY =
        'services_connector/services_connector_integration/sandbox_private_key';
    public const CONFIG_PATH_SERVICES_CONNECTOR_PRODUCTION_API_KEY =
        'services_connector/services_connector_integration/production_api_key';
    public const CONFIG_PATH_SERVICES_CONNECTOR_PRODUCTION_PRIVATE_KEY =
        'services_connector/services_connector_integration/production_private_key';

    /**
     * @var ScopeConfigInterface
     */
    private $config;

    /**
     * @var WriterInterface
     */
    private $configWriter;

    /**
     * @var TypeListInterface
     */
    protected $cacheTypeList;

    /**
     * @param ScopeConfigInterface $config
     * @param WriterInterface $configWriter
     * @param TypeListInterface $cacheTypeList
     */
    public function __construct(
        ScopeConfigInterface $config,
        WriterInterface $configWriter,
        TypeListInterface $cacheTypeList
    ) {
        $this->config = $config;
        $this->configWriter = $configWriter;
        $this->cacheTypeList = $cacheTypeList;
    }

    /**
     * @inheritdoc
     */
    public function getProjectId() : ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_PROJECT_ID);
    }

    /**
     * @inheritDoc
     */
    public function getProjectName(): ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_PROJECT_NAME);
    }

    /**
     * @inheritDoc
     */
    public function getEnvironmentId() : ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_ENVIRONMENT_ID);
    }

    /**
     * @inheritDoc
     */
    public function getEnvironmentName(): ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_ENVIRONMENT_NAME);
    }

    /**
     * @inheritdoc
     */
    public function getEnvironmentType() : ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_ENVIRONMENT_TYPE);
    }

    /**
     * @inheritDoc
     */
    public function getSandboxApiKey(): ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_SERVICES_CONNECTOR_SANDBOX_API_KEY);
    }

    /**
     * @inheritDoc
     */
    public function getSandboxPrivateKey(): ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_SERVICES_CONNECTOR_SANDBOX_PRIVATE_KEY);
    }

    /**
     * @inheritDoc
     */
    public function getProductionApiKey(): ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_SERVICES_CONNECTOR_PRODUCTION_API_KEY);
    }

    /**
     * @inheritDoc
     */
    public function getProductionPrivateKey(): ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_SERVICES_CONNECTOR_PRODUCTION_PRIVATE_KEY);
    }

    /**
     * @inheritdoc
     */
    public function getImsOrganizationId() : ?string
    {
        return $this->config->getValue(self::CONFIG_PATH_IMS_ORGANIZATION_ID);
    }

    /**
     * @inheritDoc
     */
    public function getCloudId(): ?string
    {
        // phpcs:ignore Magento2.Security.Superglobal
        return $_ENV["MAGENTO_CLOUD_PROJECT"] ?? null;
    }

    /**
     * @inheritDoc
     */
    public function isApiKeySet() : bool
    {
        $apiKey = false;
        $privateKey = false;
        $environment = $this->config->getValue(self::CONFIG_PATH_SERVICES_CONNECTOR_ENVIRONMENT);
        if ($environment) {
            $apiKey = $this->config->getValue(str_replace(
                '{env}',
                $environment,
                self::CONFIG_PATH_SERVICES_CONNECTOR_API_KEY
            ));
            $privateKey = $this->config->getValue(str_replace(
                '{env}',
                $environment,
                self::CONFIG_PATH_SERVICES_CONNECTOR_PRIVATE_KEY
            ));
        }
        return $apiKey && $privateKey;
    }

    /**
     * @inheritDoc
     */
    public function getRegistryApiUrl(string $uri) : string
    {
        return $this->config->getValue(self::CONFIG_PATH_REGISTRY_API_PATH) . $uri;
    }

    /**
     * @inheritDoc
     */
    public function setConfigValues(array $configs) : void
    {
        foreach ($configs as $key => $value) {
            if ($value === null || $value === '') {
                $this->configWriter->delete($key);
            } else {
                $this->configWriter->save($key, $value);
            }
        }
        $this->cacheTypeList->cleanType(CacheConfig::TYPE_IDENTIFIER);
    }
}
