<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\ServiceProxy\Controller\Adminhtml\Config;

use Magento\Backend\App\Action\Context;
use Magento\Framework\App\Action\HttpGetActionInterface;
use Magento\Framework\App\Action\HttpPatchActionInterface;
use Magento\Framework\App\Cache\Type\Config as CacheConfig;
use Magento\Framework\App\Cache\TypeListInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\ServiceProxy\Controller\Adminhtml\AbstractProxyController;
use Exception;

/**
 * Config Provider
 */
class Index extends AbstractProxyController implements
    HttpGetActionInterface,
    HttpPatchActionInterface
{
    const ADMIN_RESOURCE = 'Magento_ServiceProxy::services';

    /**
     * @var configPaths
     */
    private $configPaths;

    /**
     * @var WriterInterface
     */
    private $configWriter;

    /**
     * @var JsonFactory
     */
    private $jsonFactory;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var Json
     */
    private $serializer;

    /**
     * @var TypeListInterface
     */
    private $typeList;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @param Context $context
     * @param WriterInterface $configWriter
     * @param JsonFactory $jsonFactory
     * @param ScopeConfigInterface $scopeConfig
     * @param Json $serializer
     * @param TypeListInterface $typeList
     * @param StoreManagerInterface $storeManager
     * @param array $configPaths
     */
    public function __construct(
        Context $context,
        WriterInterface $configWriter,
        JsonFactory $jsonFactory,
        ScopeConfigInterface $scopeConfig,
        Json $serializer,
        TypeListInterface $typeList,
        StoreManagerInterface $storeManager,
        array $configPaths = []
    ) {
        parent::__construct($context);
        $this->configWriter = $configWriter;
        $this->jsonFactory = $jsonFactory;
        $this->scopeConfig = $scopeConfig;
        $this->serializer = $serializer;
        $this->typeList = $typeList;
        $this->configPaths = $configPaths;
        $this->storeManager = $storeManager;
    }

    /**
     * Retrieve and update service configurations remotely
     *
     * @return ResultInterface
     */
    public function execute(): ResultInterface
    {
        $request = $this->getRequest();
        $method = $request->getMethod();
        $scope = $request->getParam('scope') ?: ScopeConfigInterface::SCOPE_TYPE_DEFAULT;
        $scopeCode = $request->getParam('scopecode') ?: null;
        $service = $request->getParam('service');
        $response = $this->jsonFactory->create();
        if (!array_key_exists($service, $this->configPaths)) {
            return $response->setHttpResponseCode(404)->setData('Service not defined');
        }
        if ($method === 'PATCH') {
            try {
                $config = $this->serializer->unserialize($request->getContent());
                $this->saveConfig($config, $service, $scope, $scopeCode);
            } catch (Exception $e) {
                $response->setHttpResponseCode(500)->setData('Failed to save configuration, ' . $e->getMessage());
            }
            return $response->setHttpResponseCode(200)->setData('Configuration updated');
        }
        return $response->setData($this->buildConfigMap($service, $scope, $scopeCode));
    }

    /**
     * Build a configuration map of provided service
     *
     * @param string $service
     * @param string $scope
     * @param string $scopeCode
     * @return array
     */
    private function buildConfigMap(string $service, string $scope, ?string $scopeCode): array
    {
        $configMap = [];
        foreach ($this->configPaths[$service] as $path) {
            $configMap[$path] = $this->scopeConfig->getValue($path, $scope, $scopeCode);
        }
        return $configMap;
    }

    /**
     * Save config
     *
     * @param array $config
     * @param string $service
     * @param string $scope
     * @param string $scopeCode
     * @return void
     */
    private function saveConfig(array $config, string $service, string $scope, ?string $scopeCode): void
    {
        $storeId = $scopeCode === null ? Store::DEFAULT_STORE_ID : $this->storeManager->getStore($scopeCode)->getId();
        foreach ($config as $configKey => $configValue) {
            if (in_array($configKey, $this->configPaths[$service])) {
                $this->configWriter->save($configKey, $configValue, $scope, $storeId);
            }
        }
        // Clear config cache
        $this->typeList->cleanType(CacheConfig::TYPE_IDENTIFIER);
    }
}
