<?php
/**
 * Copyright 2015 Adobe
 * All Rights Reserved.
 */
namespace Magento\Sales\Model\ResourceModel\Order;

use Magento\Framework\Model\AbstractModel;
use Magento\Sales\Model\ResourceModel\EntityAbstract as SalesResource;
use Magento\Sales\Model\Spi\OrderAddressResourceInterface;
use Magento\Framework\Model\ResourceModel\Db\VersionControl\Snapshot;

/**
 * Flat sales order address resource
 */
class Address extends SalesResource implements OrderAddressResourceInterface
{
    /**
     * Sales order address event prefix
     *
     * @var string
     */
    protected $_eventPrefix = 'sales_order_address_resource';

    /**
     * @var \Magento\Sales\Model\Order\Address\Validator
     */
    protected $_validator;

    /**
     * @var \Magento\Sales\Model\ResourceModel\GridPool
     */
    protected $gridPool;

    /**
     * @param \Magento\Framework\Model\ResourceModel\Db\Context $context
     * @param Snapshot $entitySnapshot
     * @param \Magento\Framework\Model\ResourceModel\Db\VersionControl\RelationComposite $entityRelationComposite
     * @param \Magento\Sales\Model\ResourceModel\Attribute $attribute
     * @param \Magento\SalesSequence\Model\Manager $sequenceManager
     * @param \Magento\Sales\Model\Order\Address\Validator $validator
     * @param \Magento\Sales\Model\ResourceModel\GridPool $gridPool
     * @param string $connectionName
     */
    public function __construct(
        \Magento\Framework\Model\ResourceModel\Db\Context $context,
        Snapshot $entitySnapshot,
        \Magento\Framework\Model\ResourceModel\Db\VersionControl\RelationComposite $entityRelationComposite,
        \Magento\Sales\Model\ResourceModel\Attribute $attribute,
        \Magento\SalesSequence\Model\Manager $sequenceManager,
        \Magento\Sales\Model\Order\Address\Validator $validator,
        \Magento\Sales\Model\ResourceModel\GridPool $gridPool,
        $connectionName = null
    ) {
        $this->_validator = $validator;
        $this->gridPool = $gridPool;
        parent::__construct(
            $context,
            $entitySnapshot,
            $entityRelationComposite,
            $attribute,
            $sequenceManager,
            $connectionName
        );
    }

    /**
     * Resource initialization
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init('sales_order_address', 'entity_id');
    }

    /**
     * Return configuration for all attributes
     *
     * @return array
     */
    public function getAllAttributes()
    {
        $attributes = [
            'city' => __('City'),
            'company' => __('Company'),
            'country_id' => __('Country'),
            'email' => __('Email'),
            'firstname' => __('First Name'),
            'lastname' => __('Last Name'),
            'region_id' => __('State/Province'),
            'street' => __('Street Address'),
            'telephone' => __('Phone Number'),
            'postcode' => __('Zip/Postal Code'),
        ];
        asort($attributes);
        return $attributes;
    }

    /**
     * Performs validation before save
     *
     * @param \Magento\Framework\Model\AbstractModel $object
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function _beforeSave(\Magento\Framework\Model\AbstractModel $object)
    {
        parent::_beforeSave($object);
        if (!$object->getParentId() && $object->getOrder()) {
            $object->setParentId($object->getOrder()->getId());
        }
        // Init customer address id if customer address is assigned
        $customerData = $object->getCustomerAddressData();
        if ($customerData) {
            $object->setCustomerAddressId($customerData->getId());
        }
        $warnings = $this->_validator->validate($object);
        if (!empty($warnings)) {
            throw new \Magento\Framework\Exception\LocalizedException(
                __("We can't save the address:\n%1", implode("\n", $warnings))
            );
        }
        return $this;
    }

    /**
     * @inheritdoc
     */
    protected function isModified(AbstractModel $entity): bool
    {
        if (!$entity->getId()) {
            return true;
        }
        $snapShotData = $this->entitySnapshot->getSnapshotData($entity);
        foreach ($snapShotData as $field => $value) {
            $fieldValue = $entity->getDataByKey($field);
            if (is_numeric($fieldValue) && is_numeric($value)) {
                if ($fieldValue !== $value) {
                    return true;
                }
            }
        }
        return parent::isModified($entity);
    }
}
