<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\SaaSSales\Test\Integration;

use Magento\DataExporter\Model\FeedPool;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\SaaSCommon\Cron\SubmitFeedInterface;
use Magento\SaaSCommon\Model\Http\Command\SubmitFeed as SubmitFeedCommand;
use Magento\SaaSCommon\Test\Integration\SubmitFeedStub;
use Magento\SaaSSales\Cron\OrdersSubmitFeed;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\OrderInterfaceFactory;
use Magento\SalesOrdersDataExporter\Test\Integration\AbstractOrderFeedTest;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestModuleDataSentOutside\Model\AccessedData;

/**
 * Test class to check sales orders submit feed functionality
 *
 * @SuppressWarnings(PHPMD.UnusedPrivateMethod)
 */
class SubmitOrdersFeedTest extends AbstractOrderFeedTest
{
    /**
     * @var OrderInterfaceFactory
     */
    private $orderFactory;

    /**
     * @var SubmitFeedInterface
     */
    private $submitFeed;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    public static function setUpBeforeClass(): void
    {
        parent::setUpBeforeClass();
        Bootstrap::getObjectManager()->configure([
            'preferences' => [
                SubmitFeedCommand::class =>
                    SubmitFeedStub::class,
            ]
        ]);
        $connection = Bootstrap::getObjectManager()->create(ResourceConnection::class)->getConnection();
        $feedTable = $connection->getTableName(
            Bootstrap::getObjectManager()->get(FeedPool::class)
                ->getFeed('orders')
                ->getFeedMetadata()
                ->getFeedTableName()
        );
        $connection->truncateTable($feedTable);
    }

    /**
     * Integration test setup
     */
    protected function setUp(): void
    {
        parent::setUp();
        $objectManager = Bootstrap::getObjectManager();
        $this->orderFactory = $objectManager->get(OrderInterfaceFactory::class);
        $this->submitFeed = $objectManager->get(OrdersSubmitFeed::class); // @phpstan-ignore-line
        $this->serializer = $objectManager->create(SerializerInterface::class);
    }

    /**
     * @magentoDbIsolation disabled
     * @magentoAppIsolation enabled
     * @magentoConfigFixture magento_saas/environment production
     * @magentoConfigFixture services_connector/services_connector_integration/production_api_key test_key
     * @magentoConfigFixture services_connector/services_connector_integration/production_private_key private_test_key
     * @magentoDataFixture Magento/Sales/_files/customer_order_with_taxable_product.php
     *
     * @return void
     * @throws \Zend_Db_Statement_Exception
     */
    public function testLogDataSentOutside() : void
    {
        $expectedData = [
            'event_id' => 'magento_logging_event_accessed_data',
            'event_code' => 'inter_service_communication',
            'action' => 'data_sent_outside',
            'info' => 'sales',
            'is_success' => true,
            'user' => 'Commerce Data Exporter',
            'fullaction' => 'comdomainsvc-order-service'
        ];
        /** @var OrderInterface $order */
        $order = $this->orderFactory->create()->loadByIncrementId('test_order_with_taxable_product');
        $orderId = $order->getEntityId();
        $this->runIndexer([$orderId]);

        $this->submitFeed->execute();
        $accessedData = Bootstrap::getObjectManager()->get(AccessedData::class);
        $orderFeed = $this->getOrderFeedByIds([$orderId])[0];
        $eventData = $accessedData->getData('event');
        self::assertNotEmpty($eventData, 'Event data is empty');
        $accessedDataLogged = $this->serializer->unserialize($eventData);
        foreach ($expectedData as $expectedField => $expectedValue) {
            self::assertArrayHasKey($expectedField, $accessedDataLogged);
            self::assertEquals($expectedValue, $accessedDataLogged[$expectedField]);
        }
        $loggedInfo = $accessedDataLogged['data'][0] ?? null;
        self::assertNotEmpty($loggedInfo);
        self::assertEquals(
            $orderFeed,
            $loggedInfo
        );
    }
}
